/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.loader.java.validation;

import static org.mule.runtime.extension.api.util.ExtensionMetadataTypeUtils.getType;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.getImplementingName;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.isInstantiable;

import static java.lang.String.format;
import static java.lang.String.join;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.parameter.ActingParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.parameter.ValueProviderModel;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.extension.api.loader.ExtensionModelValidator;
import org.mule.runtime.extension.api.loader.Problem;
import org.mule.runtime.extension.api.loader.ProblemsReporter;
import org.mule.runtime.module.extension.internal.loader.java.property.ValueProviderFactoryModelProperty;
import org.mule.runtime.module.extension.internal.loader.parser.java.JavaValueProviderFactory;
import org.mule.runtime.module.extension.internal.loader.validator.AbstractValueProviderModelValidator;
import org.mule.runtime.module.extension.internal.loader.validator.AbstractValueProviderModelValidator.ValidationContext;
import org.mule.runtime.module.extension.internal.loader.validator.ValueProviderModelValidator;
import org.mule.runtime.module.extension.internal.util.ReflectionCache;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * {@link ExtensionModelValidator} for the correct usage of {@link ValueProviderModel} and
 * {@link ValueProviderFactoryModelProperty}. This validator has validations specific to the Java SDK.
 *
 * @since 4.0
 * @see ValueProviderModelValidator
 */
public final class JavaValueProviderModelValidator
    extends AbstractValueProviderModelValidator<JavaValueProviderModelValidator.JavaValueProviderModelValidationContext> {

  @Override
  public void validate(ExtensionModel model, ProblemsReporter problemsReporter) {
    var valueProvidersIdValidator = new ValueProvidersIdValidator(problemsReporter);
    doValidate(model, problemsReporter,
               new JavaValueProviderModelValidationContext(new ReflectionCache(), valueProvidersIdValidator));
    valueProvidersIdValidator.validateIdsAreUnique();
  }

  @Override
  protected String getProviderId(ValueProviderFactoryModelProperty modelProperty, ValueProviderModel valueProviderModel) {
    JavaValueProviderFactory javaValueProviderFactory = (JavaValueProviderFactory) modelProperty.getFactory();
    Class<?> valueProvider = javaValueProviderFactory.getResolverClass();
    return valueProvider.getSimpleName();
  }


  @Override
  protected void doValidateProvider(ParameterizedModel containerModel, ValueProviderFactoryModelProperty modelProperty,
                                    ValueProviderModel valueProviderModel, ProblemsReporter problemsReporter,
                                    JavaValueProviderModelValidationContext validationContext) {
    super.doValidateProvider(containerModel, modelProperty, valueProviderModel, problemsReporter, validationContext);

    JavaValueProviderFactory javaValueProviderFactory = (JavaValueProviderFactory) modelProperty.getFactory();
    Class<?> valueProvider = javaValueProviderFactory.getResolverClass();
    String providerName = valueProvider.getSimpleName();

    validationContext.valueProvidersIdValidator
        .addValueProviderInformation(new ValueProviderInformation(valueProviderModel, containerModel, valueProvider.getName()));
    if (!isInstantiable(valueProvider, validationContext.reflectionCache)) {
      problemsReporter.addError(new Problem(containerModel, format("The Value Provider [%s] is not instantiable but it should",
                                                                   providerName)));
    }
  }

  @Override
  protected void doValidateInjectableParameters(ParameterizedModel containerModel,
                                                String containerName, String containerTypeName,
                                                Map<String, MetadataType> containerParameterTypesByName,
                                                ValueProviderFactoryModelProperty modelProperty,
                                                ValueProviderModel valueProviderModel,
                                                ProblemsReporter problemsReporter,
                                                JavaValueProviderModelValidationContext validationContext) {
    JavaValueProviderFactory javaValueProviderFactory = (JavaValueProviderFactory) modelProperty.getFactory();
    Class<?> valueProvider = javaValueProviderFactory.getResolverClass();
    String providerName = valueProvider.getSimpleName();

    Map<String, String> containerParameterAliases = getContainerParameterNames(containerModel.getAllParameterModels());
    Map<String, MetadataType> targetParameterTypesByName = javaValueProviderFactory.getFieldTypesByName();
    for (ActingParameterModel parameterInfo : valueProviderModel.getParameters()) {
      String parameterName = containerParameterAliases.get(parameterInfo.getName());
      if (parameterInfo.getExtractionExpression().equals(parameterName)) {
        MetadataType metadataType = containerParameterTypesByName.get(parameterName);
        Class<?> expectedType = getType(metadataType)
            .orElseThrow(() -> new IllegalStateException(format("Unable to get Class for parameter: %s", parameterName)));
        Class<?> gotType = getType(targetParameterTypesByName.get(parameterInfo.getName()))
            .orElseThrow(() -> new IllegalStateException(format("Unable to get Class for parameter: %s", parameterName)));

        if (!expectedType.equals(gotType)) {
          problemsReporter.addError(new Problem(containerModel,
                                                format("The Value Provider [%s] defines a parameter '%s' of type '%s' but in the %s '%s' is of type '%s'",
                                                       providerName, parameterName, gotType, containerTypeName,
                                                       containerName, expectedType)));
        }
      }
    }
  }

  private Map<String, String> getContainerParameterNames(List<ParameterModel> allParameters) {
    Map<String, String> parameterNames = new HashMap<>();
    for (ParameterModel parameterDeclaration : allParameters) {
      parameterNames.put(getImplementingName(parameterDeclaration), parameterDeclaration.getName());
    }
    return parameterNames;
  }

  private static final class ValueProviderInformation {

    private final ValueProviderModel valueProviderModel;
    private final ParameterizedModel ownerModel;
    private final String implementationClassName;

    public ValueProviderInformation(ValueProviderModel valueProviderModel,
                                    ParameterizedModel ownerModel, String implementationClassName) {
      this.valueProviderModel = valueProviderModel;
      this.ownerModel = ownerModel;
      this.implementationClassName = implementationClassName;
    }

    public ValueProviderModel getValueProviderModel() {
      return valueProviderModel;
    }

    public ParameterizedModel getOwnerModel() {
      return ownerModel;
    }

    public String getImplementationClassName() {
      return implementationClassName;
    }
  }

  private static final class ValueProvidersIdValidator {

    private final Map<String, ValueProviderInformation> valueProvidersImplementationToInformation = new HashMap<>();
    private final MultiMap<String, String> valueProvidersIdToImplementations = new MultiMap<>();
    private final ProblemsReporter problemsReporter;

    public ValueProvidersIdValidator(ProblemsReporter problemsReporter) {
      this.problemsReporter = problemsReporter;
    }

    public void addValueProviderInformation(ValueProviderInformation valueProviderInformation) {
      String valueProviderImplementation = valueProviderInformation.getImplementationClassName();
      if (!valueProvidersImplementationToInformation.containsKey(valueProviderImplementation)) {
        valueProvidersImplementationToInformation.put(valueProviderImplementation, valueProviderInformation);
        valueProvidersIdToImplementations.put(valueProviderInformation.getValueProviderModel().getProviderId(),
                                              valueProviderImplementation);
      }
    }

    public void validateIdsAreUnique() {
      valueProvidersIdToImplementations.keySet().forEach((valueProviderId) -> {
        List<String> valueProviderImplementations = valueProvidersIdToImplementations.getAll(valueProviderId);

        if (valueProviderImplementations.size() > 1) {
          String firstValueProviderImplementation = valueProviderImplementations.get(0);
          ValueProviderInformation valueProviderInformation =
              valueProvidersImplementationToInformation.get(firstValueProviderImplementation);
          problemsReporter.addError(new Problem(valueProviderInformation.getOwnerModel(),
                                                format("The following ValueProvider implementations [%s] use the same id [%s]. "
                                                    +
                                                    "ValueProvider ids must be unique.",
                                                       join(", ", valueProviderImplementations), valueProviderId)));
        }
      });
    }
  }

  public record JavaValueProviderModelValidationContext(ReflectionCache reflectionCache,
                                                        ValueProvidersIdValidator valueProvidersIdValidator)
      implements ValidationContext {

  }
}
