/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.source.legacy;

import org.mule.runtime.api.connection.ConnectionException;
import org.mule.sdk.api.runtime.operation.Result;
import org.mule.sdk.api.runtime.source.PollContext;
import org.mule.sdk.api.runtime.source.SourceCallbackContext;

import java.io.Serializable;
import java.util.Comparator;
import java.util.Optional;
import java.util.function.Consumer;

/**
 * Adapts a {@link org.mule.runtime.extension.api.runtime.source.PollContext} into a {@link PollContext}
 *
 * @param <T> the generic type of the output value
 * @param <A> the generic type of the message attributes
 * @since 4.10.0
 */
public class SdkPollContextAdapter<T, A> implements PollContext<T, A> {

  private final org.mule.runtime.extension.api.runtime.source.PollContext<T, A> delegate;

  public SdkPollContextAdapter(org.mule.runtime.extension.api.runtime.source.PollContext<T, A> delegate) {
    this.delegate = delegate;
  }

  @Override
  public PollItemStatus accept(Consumer<PollItem<T, A>> consumer) {
    return from(delegate.accept(new PollItemConsumerAdapter<>(consumer)));
  }

  @Override
  public Optional<Serializable> getWatermark() {
    return delegate.getWatermark();
  }

  @Override
  public boolean isSourceStopping() {
    return delegate.isSourceStopping();
  }

  @Override
  public void setWatermarkComparator(Comparator<? extends Serializable> comparator) {
    delegate.setWatermarkComparator(comparator);
  }

  @Override
  public void onConnectionException(ConnectionException e) {
    delegate.onConnectionException(e);
  }

  private static class PollItemConsumerAdapter<T, A>
      implements Consumer<org.mule.runtime.extension.api.runtime.source.PollContext.PollItem<T, A>> {

    Consumer<PollItem<T, A>> delegate;

    PollItemConsumerAdapter(Consumer<PollItem<T, A>> delegate) {
      this.delegate = delegate;
    }

    @Override
    public void accept(org.mule.runtime.extension.api.runtime.source.PollContext.PollItem<T, A> pollItem) {
      delegate.accept(new LegacyToSdkPollItemAdapter<>(pollItem));
    }

    private static class LegacyToSdkPollItemAdapter<T, A> implements PollItem<T, A> {

      org.mule.runtime.extension.api.runtime.source.PollContext.PollItem<T, A> delegate;

      public LegacyToSdkPollItemAdapter(org.mule.runtime.extension.api.runtime.source.PollContext.PollItem<T, A> delegate) {
        this.delegate = delegate;
      }

      @Override
      public SourceCallbackContext getSourceCallbackContext() {
        return new SdkSourceCallBackContextAdapter(delegate.getSourceCallbackContext());
      }

      @Override
      public PollItem<T, A> setResult(Result result) {
        delegate.setResult(LegacyResultAdapter.from(result));
        return this;
      }

      @Override
      public PollItem<T, A> setWatermark(Serializable watermark) {
        delegate.setWatermark(watermark);
        return this;
      }

      @Override
      public PollItem<T, A> setId(String id) {
        delegate.setId(id);
        return this;
      }
    }
  }

  public static PollItemStatus from(org.mule.runtime.extension.api.runtime.source.PollContext.PollItemStatus pollItemStatus) {
    return switch (pollItemStatus) {
      case SOURCE_STOPPING -> PollItemStatus.SOURCE_STOPPING;
      case ALREADY_IN_PROCESS -> PollItemStatus.ALREADY_IN_PROCESS;
      case FILTERED_BY_WATERMARK -> PollItemStatus.FILTERED_BY_WATERMARK;
      case ACCEPTED -> PollItemStatus.ACCEPTED;
    };
  }
}
