/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.operation;

import static org.mule.runtime.module.extension.internal.runtime.connectivity.oauth.ExtensionsOAuthUtils.MAX_REFRESH_ATTEMPTS;
import static org.mule.runtime.module.extension.internal.runtime.connectivity.oauth.ExtensionsOAuthUtils.refreshTokenIfNecessary;
import static org.mule.runtime.module.extension.internal.runtime.streaming.CursorResetInterceptor.CURSOR_RESET_HANDLER_VARIABLE;

import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.extension.api.runtime.operation.CompletableComponentExecutor.ExecutorCallback;
import org.mule.runtime.module.extension.api.runtime.privileged.ExecutionContextAdapter;
import org.mule.runtime.module.extension.internal.runtime.streaming.CursorResetHandler;
import org.mule.sdk.api.connectivity.oauth.AccessTokenExpiredException;

import java.util.function.Consumer;

/**
 * An {@link ExecutorCallback} decorator that automatically refreshes OAuth access tokens when an
 * {@link AccessTokenExpiredException} is found
 *
 * @since 4.9.8, 4.10.0
 */
public class RefreshOAuthTokenExecutorCallbackDecorator implements ExecutorCallback {

  private final ExecutorCallback delegate;
  private final ExecutionContextAdapter operationContext;
  private final Consumer<ExecutorCallback> retryCommand;
  private int attempts;

  public RefreshOAuthTokenExecutorCallbackDecorator(ExecutorCallback delegate,
                                                    ExecutionContextAdapter operationContext,
                                                    Consumer<ExecutorCallback> retryCommand) {
    this.delegate = delegate;
    this.operationContext = operationContext;
    this.retryCommand = retryCommand;
    attempts = 0;
  }

  @Override
  public void complete(Object value) {
    delegate.complete(value);
  }

  @Override
  public void error(Throwable e) {
    try {
      if (++attempts <= MAX_REFRESH_ATTEMPTS && refreshTokenIfNecessary(operationContext, e)) {
        resetCursors(operationContext);
        retryCommand.accept(this);
      } else {
        delegate.error(e);
      }
    } catch (Exception refreshException) {
      delegate.error(refreshException);
    }
  }

  private void resetCursors(ExecutionContextAdapter<OperationModel> operationContext) {
    CursorResetHandler cursorResetHandler =
        operationContext.getVariable(CURSOR_RESET_HANDLER_VARIABLE);
    if (cursorResetHandler != null) {
      cursorResetHandler.resetCursors();
    }
  }
}
