/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.resolver;

import static org.mule.test.module.extension.internal.util.ExtensionsTestUtils.getParameter;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.MatcherAssert.assertThat;

import static org.junit.jupiter.api.Assertions.assertThrows;

import static org.mockito.Mockito.when;

import org.mule.runtime.api.lifecycle.Lifecycle;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.context.MuleContextAware;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.module.extension.api.runtime.resolver.ResolverSet;
import org.mule.runtime.module.extension.api.runtime.resolver.ResolverSetResult;
import org.mule.runtime.module.extension.api.runtime.resolver.ValueResolver;
import org.mule.runtime.module.extension.api.runtime.resolver.ValueResolvingContext;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;
import org.mule.test.module.extension.internal.util.ExtensionsTestUtils;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

@SmallTest
@ExtendWith(MockitoExtension.class)
public class ResolverSetTestCase extends AbstractMuleTestCase {

  private static final String NAME = "MG";
  private static final int AGE = 31;

  private ResolverSet set;
  private Map<ParameterModel, ValueResolver> mapping;

  @Mock
  private CoreEvent event;

  @Mock(lenient = true)
  private ValueResolvingContext resolvingContext;

  @Mock
  private MuleContext muleContext;

  @BeforeEach
  public void before() throws Exception {
    mapping = new LinkedHashMap<>();
    mapping.put(getParameter("myName", String.class), getResolver(NAME));
    mapping.put(getParameter("age", Integer.class), getResolver(AGE));

    when(resolvingContext.getEvent()).thenReturn(event);
    when(resolvingContext.getConfig()).thenReturn(Optional.empty());

    set = buildSet(mapping);
  }

  @Test
  void resolve() throws Exception {
    ResolverSetResult result = set.resolve(resolvingContext);
    assertResult(result, mapping);
  }

  @Test
  void addNullParameter() throws Exception {
    final ValueResolver nullResolver = getResolver(null);
    assertThrows(IllegalArgumentException.class, () -> set.add(null, nullResolver));
  }

  @Test
  void addNullresolver() throws Exception {
    assertThrows(NullPointerException.class, () -> set.add("blah", null));
  }

  @Test
  void addRepeatedParameter() throws Exception {
    final String parameterName = "name";
    ValueResolver<String> resolver = getResolver(NAME);

    set.add(parameterName, resolver);
    assertThrows(IllegalStateException.class, () -> set.add(parameterName, resolver));
  }

  @Test
  void isNotDynamic() {
    assertThat(set.isDynamic(), is(false));
  }

  @Test
  void isDynamic() throws Exception {
    ValueResolver resolver = getResolver(null);
    when(resolver.isDynamic()).thenReturn(true);

    set.add("whatever", resolver);
    assertThat(set.isDynamic(), is(true));
  }

  private void assertResult(ResolverSetResult result, Map<ParameterModel, ValueResolver> mapping) throws Exception {
    assertThat(result, is(notNullValue()));
    for (Map.Entry<ParameterModel, ValueResolver> entry : mapping.entrySet()) {
      Object value = result.get(entry.getKey().getName());
      assertThat(value, is(entry.getValue().resolve(resolvingContext)));
    }
  }

  private ResolverSet buildSet(Map<ParameterModel, ValueResolver> mapping) {
    ResolverSet set = new ResolverSet(muleContext);
    mapping.forEach((key, value) -> set.add(key.getName(), value));

    return set;
  }

  private ValueResolver getResolver(Object value) throws Exception {
    return ExtensionsTestUtils.getResolver(value, resolvingContext, false, MuleContextAware.class, Lifecycle.class);
  }
}
