/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.loader.parser.java;

import static org.mule.runtime.module.extension.internal.loader.parser.java.MuleExtensionAnnotationParser.mapReduceSingleAnnotation;
import static org.mule.runtime.module.extension.internal.loader.utils.ParameterUtils.getConfigFields;
import static org.mule.runtime.module.extension.internal.loader.utils.ParameterUtils.getConnectionFields;
import static org.mule.runtime.module.extension.internal.loader.utils.ParameterUtils.getParameterFields;
import static org.mule.runtime.module.extension.internal.value.ValueProviderUtils.getValueProviderId;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toMap;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.extension.api.annotation.values.OfValues;
import org.mule.runtime.extension.api.annotation.values.ValuePart;
import org.mule.runtime.extension.api.loader.parser.ActingParameterModelParser;
import org.mule.runtime.extension.api.loader.parser.ValueProviderFactory;
import org.mule.runtime.extension.api.loader.parser.ValueProviderModelParser;
import org.mule.runtime.extension.api.property.SinceMuleVersionModelProperty;
import org.mule.runtime.module.extension.api.loader.java.type.AnnotationValueFetcher;
import org.mule.runtime.module.extension.api.loader.java.type.ExtensionParameter;
import org.mule.runtime.module.extension.api.loader.java.type.FieldElement;
import org.mule.runtime.module.extension.internal.loader.java.type.runtime.ParameterizableTypeWrapper;
import org.mule.sdk.api.annotation.binding.Binding;

import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * {@link ValueProviderModelParser} for Java based syntax
 *
 * @since 4.10.0
 * @implNote Logic was extracted from what used to be the ValueProvidersParameterDeclarationEnricher in earlier versions.
 */
class JavaValueProviderModelParser implements ValueProviderModelParser {

  private static final SinceMuleVersionModelProperty SINCE_MULE_VERSION_MODEL_PROPERTY_SDK_API_VP =
      new SinceMuleVersionModelProperty("4.4.0");

  public static Optional<JavaValueProviderModelParser> parseFromOfValues(ExtensionParameter ofValuesSource,
                                                                         ExtensionParameter valuePartsSource,
                                                                         String elementType,
                                                                         ClassTypeLoader typeLoader) {
    return mapReduceSingleAnnotation(ofValuesSource,
                                     elementType,
                                     ofValuesSource.getName(),
                                     OfValues.class,
                                     org.mule.sdk.api.annotation.values.OfValues.class,
                                     fetcher -> getFromOfValuesFetcherLegacy(valuePartsSource, fetcher, typeLoader),
                                     fetcher -> getFromOfValuesFetcher(valuePartsSource, fetcher, typeLoader))
                                         .filter(ofValueInformation -> ofValueInformation.clazz != null);
  }

  private static JavaValueProviderModelParser getFromOfValuesFetcherLegacy(ExtensionParameter valuePartsSource,
                                                                           AnnotationValueFetcher<OfValues> fetcher,
                                                                           ClassTypeLoader typeLoader) {
    return new JavaValueProviderModelParser(fetcher.getClassValue(OfValues::value).getDeclaringClass().orElse(null),
                                            fetcher.getBooleanValue(OfValues::open),
                                            emptyMap(),
                                            getPartOrderFromValueParts(valuePartsSource),
                                            typeLoader,
                                            true);
  }

  private static JavaValueProviderModelParser getFromOfValuesFetcher(ExtensionParameter valuePartsSource,
                                                                     AnnotationValueFetcher<org.mule.sdk.api.annotation.values.OfValues> fetcher,
                                                                     ClassTypeLoader typeLoader) {
    return new JavaValueProviderModelParser(fetcher.getClassValue(org.mule.sdk.api.annotation.values.OfValues::value)
        .getDeclaringClass().orElse(null),
                                            fetcher.getBooleanValue(org.mule.sdk.api.annotation.values.OfValues::open),
                                            getBindingsMapFromFetcher(fetcher
                                                .getInnerAnnotations(org.mule.sdk.api.annotation.values.OfValues::bindings)),
                                            getPartOrderFromValueParts(valuePartsSource),
                                            typeLoader,
                                            false);
  }

  private static Optional<Integer> getPartOrderFromValueParts(ExtensionParameter valuePartsSource) {
    return mapReduceSingleAnnotation(valuePartsSource,
                                     "parameter",
                                     valuePartsSource.getName(),
                                     ValuePart.class,
                                     org.mule.sdk.api.annotation.values.ValuePart.class,
                                     fetcher -> fetcher.getNumberValue(ValuePart::order),
                                     fetcher -> fetcher.getNumberValue(org.mule.sdk.api.annotation.values.ValuePart::order));
  }

  private static Map<String, String> getBindingsMapFromFetcher(List<AnnotationValueFetcher<Binding>> annotationValueFetcher) {
    return annotationValueFetcher.stream()
        .collect(toMap(fetcher -> fetcher.getStringValue(Binding::actingParameter),
                       fetcher -> fetcher.getStringValue(Binding::extractionExpression)));
  }

  private final Class<?> clazz;
  private final boolean isOpen;
  private final Optional<Integer> partOrder;
  private final boolean fromLegacyAnnotation;

  private final List<FieldElement> providerParameters;
  private final List<ActingParameterModelParser> actingParameterModelParsers;
  private final List<FieldElement> configFields;
  private final List<FieldElement> connectionFields;

  protected JavaValueProviderModelParser(Class<?> clazz,
                                         boolean isOpen,
                                         Map<String, String> bindingsMap,
                                         Optional<Integer> partOrder,
                                         ClassTypeLoader typeLoader,
                                         boolean fromLegacyAnnotation) {
    this.clazz = clazz;
    this.isOpen = isOpen;
    this.partOrder = partOrder;
    this.fromLegacyAnnotation = fromLegacyAnnotation;

    ParameterizableTypeWrapper providerClassWrapper = new ParameterizableTypeWrapper(clazz, typeLoader);
    this.providerParameters = getParameterFields(providerClassWrapper);
    this.configFields = getConfigFields(providerClassWrapper);
    this.connectionFields = getConnectionFields(providerClassWrapper);

    actingParameterModelParsers = providerParameters.stream()
        .map(fieldElement -> getActingParameterModelParser(fieldElement, bindingsMap))
        .toList();
  }

  @Override
  public String getId() {
    return getValueProviderId(clazz);
  }

  @Override
  public boolean isOpen() {
    return isOpen;
  }

  @Override
  public List<ActingParameterModelParser> getActingParameterParsers() {
    return actingParameterModelParsers;
  }

  @Override
  public boolean requiresConfiguration() {
    return !configFields.isEmpty();
  }

  @Override
  public boolean requiresConnection() {
    return !connectionFields.isEmpty();
  }

  @Override
  public Optional<Integer> getPartOrder() {
    return partOrder;
  }

  @Override
  public ValueProviderFactory getValueProviderFactory() {
    // Only one parameter can be annotated with @Config and only one with @Connection.
    // We could relax that restriction, but we would be breaking backwards
    return new JavaValueProviderFactory(clazz, providerParameters,
                                        configFields.isEmpty() ? null : configFields.get(0),
                                        connectionFields.isEmpty() ? null : connectionFields.get(0));
  }

  private ActingParameterModelParser getActingParameterModelParser(FieldElement fieldElement, Map<String, String> bindingsMap) {
    return new JavaActingParameterModelParser(fieldElement.getName(),
                                              fieldElement.isRequired(),
                                              bindingsMap.get(fieldElement.getName()));
  }

  @Override
  public List<ModelProperty> getAdditionalModelProperties() {
    return fromLegacyAnnotation ? emptyList() : singletonList(SINCE_MULE_VERSION_MODEL_PROPERTY_SDK_API_VP);
  }
}
