/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.store;

import static org.mule.runtime.core.api.config.MuleProperties.LOCAL_OBJECT_STORE_MANAGER;
import static org.mule.runtime.module.extension.internal.store.SdkObjectStoreUtils.convertToMuleObjectStoreSettings;

import static java.util.Collections.synchronizedSet;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.runtime.api.store.ObjectStoreManager;
import org.mule.sdk.api.store.ObjectStore;
import org.mule.sdk.api.store.ObjectStoreSettings;

import java.io.Serializable;
import java.util.HashSet;
import java.util.Set;
import java.util.function.BiFunction;

import org.slf4j.Logger;

import jakarta.inject.Inject;
import jakarta.inject.Named;

/**
 * Adapts the Mule api for object store manager {@link ObjectStoreManager} into the SDK api
 * {@link org.mule.sdk.api.store.ObjectStoreManager}
 *
 * @since 4.5.0
 */
public class SdkObjectStoreManagerAdapter implements org.mule.sdk.api.store.ObjectStoreManager {

  private static final Logger LOGGER = getLogger(SdkObjectStoreManagerAdapter.class);

  private ObjectStoreManager delegate;

  private Set<String> localStoreNames = synchronizedSet(new HashSet<>());
  private ObjectStoreManager localDelegate;

  @Inject
  public void setDelegate(ObjectStoreManager delegate) {
    this.delegate = delegate;
  }

  @Inject
  @Named(LOCAL_OBJECT_STORE_MANAGER)
  public void setLocalDelegate(ObjectStoreManager localDelegate) {
    this.localDelegate = localDelegate;
  }

  @Override
  public <T extends ObjectStore<? extends Serializable>> T getObjectStore(String name) {
    org.mule.runtime.api.store.ObjectStore<? extends Serializable> objectStore;

    if (localStoreNames.contains(name)) {
      objectStore = localDelegate.getObjectStore(name);
    } else {
      objectStore = delegate.getObjectStore(name);
    }

    return (T) SdkObjectStoreAdapter.from(objectStore);
  }

  @Override
  public <T extends ObjectStore<? extends Serializable>> T createObjectStore(String name,
                                                                             ObjectStoreSettings objectStoreSettings) {
    return doCreateObjectStore(name, objectStoreSettings,
                               (d, convertedSettings) -> d.createObjectStore(name, convertedSettings));
  }

  @Override
  public <T extends ObjectStore<? extends Serializable>> T getOrCreateObjectStore(String name,
                                                                                  ObjectStoreSettings objectStoreSettings) {

    return doCreateObjectStore(name, objectStoreSettings,
                               (d, convertedSettings) -> d.getOrCreateObjectStore(name, convertedSettings));
  }

  private <T extends ObjectStore<? extends Serializable>> T doCreateObjectStore(String name,
                                                                                ObjectStoreSettings objectStoreSettings,
                                                                                BiFunction<ObjectStoreManager, org.mule.runtime.api.store.ObjectStoreSettings, org.mule.runtime.api.store.ObjectStore> objectStoreFactory) {
    final var convertedSettings = convertToMuleObjectStoreSettings(objectStoreSettings);
    org.mule.runtime.api.store.ObjectStore<? extends Serializable> objectStore;

    if (objectStoreSettings.isDistributed()) {
      objectStore = objectStoreFactory.apply(delegate, convertedSettings);
    } else {
      objectStore = objectStoreFactory.apply(localDelegate, convertedSettings);
      localStoreNames.add(name);
    }

    if (objectStoreSettings.isDistributed() != objectStore.isDistributed()) {
      LOGGER.warn("Requested a {} object store, but '{}' is {}",
                  objectStoreSettings.isDistributed() ? "distributed" : "local",
                  objectStore,
                  objectStore.isDistributed() ? "distributed" : "local");
    }
    if (objectStoreSettings.isPersistent() != objectStore.isPersistent()) {
      LOGGER.warn("Requested a {} object store, but '{}' is {}",
                  objectStoreSettings.isPersistent() ? "persistent" : "transient",
                  objectStore,
                  objectStore.isPersistent() ? "persistent" : "transient");
    }

    return (T) SdkObjectStoreAdapter.from(objectStore);
  }

  @Override
  public void disposeStore(String name) throws ObjectStoreException {
    if (localStoreNames.remove(name)) {
      localDelegate.disposeStore(name);
    } else {
      delegate.disposeStore(name);
    }
  }
}

