/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.value;

import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;

import static java.util.Collections.unmodifiableSet;
import static java.util.stream.Collectors.toCollection;

import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.context.MuleContextAware;
import org.mule.runtime.extension.api.values.ValueProvider;
import org.mule.runtime.extension.api.values.ValueResolvingException;

import java.util.LinkedHashSet;
import java.util.Set;

/**
 * Adapter of the legacy {@link org.mule.runtime.extension.api.values.ValueProvider} to {@link ValueProvider}
 *
 * @since 4.4.0
 */
public class SdkValueProviderAdapter implements ValueProvider, MuleContextAware, Initialisable {

  private final org.mule.sdk.api.values.ValueProvider valueProvider;
  private MuleContext muleContext;

  public SdkValueProviderAdapter(org.mule.sdk.api.values.ValueProvider valueProvider) {
    this.valueProvider = valueProvider;
  }

  @Override
  public Set<Value> resolve() throws ValueResolvingException {
    try {
      Set<Value> values = valueProvider.resolve()
          .stream().map(SdkValueAdapter::new)
          .collect(toCollection(LinkedHashSet::new));
      return unmodifiableSet(values);
    } catch (org.mule.sdk.api.values.ValueResolvingException e) {
      throw new ValueResolvingException(e.getMessage(), e.getFailureCode(), e.getCause());
    }
  }

  @Override
  public String getId() {
    return valueProvider.getId();
  }

  @Override
  public void setMuleContext(MuleContext context) {
    this.muleContext = context;
  }

  @Override
  public void initialise() throws InitialisationException {
    // We need to use the version with MuleContext because these objects can be injected the MuleContext itself, and we need
    // to retain that for compatibility
    initialiseIfNeeded(valueProvider, muleContext);
  }
}
