/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.source;

import static java.lang.Integer.MAX_VALUE;
import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.extension.api.ExtensionConstants.POLLING_SOURCE_LIMIT_PARAMETER_NAME;
import static org.mule.runtime.extension.api.ExtensionConstants.SCHEDULING_STRATEGY_PARAMETER_NAME;
import static org.mule.runtime.module.extension.internal.util.MuleExtensionUtils.getSdkSourceFactory;
import static org.mule.runtime.module.extension.internal.util.MuleExtensionUtils.toBackPressureAction;

import org.mule.runtime.api.component.Component;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.exception.DefaultMuleException;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.scheduler.SchedulingStrategy;
import org.mule.runtime.core.api.el.ExpressionManager;
import org.mule.runtime.core.api.exception.SystemExceptionHandler;
import org.mule.runtime.core.api.source.MessageSource.BackPressureStrategy;
import org.mule.runtime.core.api.source.scheduler.FixedFrequencyScheduler;
import org.mule.runtime.core.api.streaming.CursorProviderFactory;
import org.mule.runtime.extension.api.runtime.config.ConfigurationInstance;
import org.mule.runtime.extension.api.runtime.source.BackPressureAction;
import org.mule.runtime.extension.api.runtime.source.SourceFactoryContext;
import org.mule.runtime.extension.api.runtime.source.SdkSourceFactory;
import org.mule.runtime.module.extension.api.runtime.resolver.ResolverSet;
import org.mule.runtime.module.extension.api.runtime.resolver.ValueResolver;
import org.mule.runtime.module.extension.api.runtime.resolver.ValueResolvingContext;
import org.mule.runtime.module.extension.internal.loader.parser.java.connection.ReverseSdkConnectionProviderAdapter;
import org.mule.runtime.module.extension.internal.runtime.source.legacy.SdkSourceAdapterFactory;
import org.mule.runtime.module.extension.internal.runtime.source.legacy.SdkSourceWrapper;
import org.mule.runtime.module.extension.internal.runtime.source.poll.PollingSourceWrapper;
import org.mule.runtime.module.extension.internal.util.FieldSetter;
import org.mule.sdk.api.runtime.source.PollingSource;
import org.mule.sdk.api.runtime.source.Source;

import java.util.Optional;

/**
 * A factory for {@link SourceAdapter} instances
 */
public class SourceAdapterFactory {

  private final ExtensionModel extensionModel;
  private final SourceModel sourceModel;
  private final ResolverSet sourceParameters;
  private final ResolverSet successCallbackParameters;
  private final ResolverSet errorCallbackParameters;
  private final CursorProviderFactory<?> cursorProviderFactory;
  private final Optional<BackPressureAction> backPressureAction;
  private final ExpressionManager expressionManager;
  private final String defaultEncoding;
  private final SystemExceptionHandler exceptionHandler;

  public SourceAdapterFactory(ExtensionModel extensionModel,
                              SourceModel sourceModel,
                              ResolverSet sourceParameters,
                              ResolverSet successCallbackParameters,
                              ResolverSet errorCallbackParameters,
                              CursorProviderFactory<?> cursorProviderFactory,
                              BackPressureStrategy backPressureStrategy,
                              ExpressionManager expressionManager,
                              String defaultEncoding, SystemExceptionHandler exceptionHandler) {
    this.extensionModel = extensionModel;
    this.sourceModel = sourceModel;
    this.sourceParameters = sourceParameters;
    this.successCallbackParameters = successCallbackParameters;
    this.errorCallbackParameters = errorCallbackParameters;
    this.cursorProviderFactory = cursorProviderFactory;
    this.backPressureAction = toBackPressureAction(backPressureStrategy);
    this.expressionManager = expressionManager;
    this.defaultEncoding = defaultEncoding;
    this.exceptionHandler = exceptionHandler;
  }

  /**
   * Creates a new {@link SourceAdapter}
   *
   * @param configurationInstance an {@link Optional} {@link ConfigurationInstance} in case the source requires a config
   * @param sourceCallbackFactory a {@link SourceCallbackFactory}
   *
   * @param restarting            indicates if the creation of the adapter was triggered after by a restart
   *
   * @return a new {@link SourceAdapter}
   */
  public SourceAdapter createAdapter(Optional<ConfigurationInstance> configurationInstance,
                                     SourceCallbackFactory sourceCallbackFactory,
                                     Component component,
                                     SourceConnectionManager connectionManager,
                                     boolean restarting) {
    try {
      SdkSourceFactory sourceFactory = getSdkSourceFactory(sourceModel);
      Source<?, ?> sdkSource;

      ComponentLocation componentLocation = component.getLocation();
      Optional<ConnectionProvider<?>> connectionProvider =
          getConnectionProvider(configurationInstance, connectionManager, componentLocation);
      SourceFactoryContext context = new ResolverBasedSourceFactoryContext(sourceModel,
                                                                           componentLocation,
                                                                           sourceParameters,
                                                                           configurationInstance,
                                                                           connectionProvider,
                                                                           expressionManager,
                                                                           defaultEncoding);

      org.mule.runtime.extension.api.runtime.source.Source<?, ?> source = sourceFactory.createMessageSource(context);

      if (source instanceof SdkSourceWrapper) {
        sdkSource = ((SdkSourceWrapper) source).getDelegate();
      } else {
        sdkSource = SdkSourceAdapterFactory.createAdapter(source);
      }

      // Handles polling source wrapper if needed
      if (sdkSource instanceof PollingSource) {
        sdkSource = wrapPollingSource((PollingSource<?, ?>) sdkSource, configurationInstance, componentLocation);
      }

      return new SourceAdapter(extensionModel,
                               sourceModel,
                               sdkSource,
                               configurationInstance,
                               cursorProviderFactory,
                               sourceCallbackFactory,
                               component,
                               sourceParameters,
                               successCallbackParameters,
                               errorCallbackParameters,
                               backPressureAction);
    } catch (Exception e) {
      throw new MuleRuntimeException(createStaticMessage(format("Could not create generator for source '%s'",
                                                                sourceModel.getName())),
                                     e);
    }
  }

  private Source<?, ?> wrapPollingSource(PollingSource<?, ?> pollingSource, Optional<ConfigurationInstance> configurationInstance,
                                         ComponentLocation componentLocation)
      throws MuleException {
    try (ValueResolvingContext context = ValueResolvingContext.builder(null, expressionManager)
        .withConfig(configurationInstance)
        .build()) {

      SchedulingStrategy scheduler = resolveSchedulingStrategy(context);
      int maxItemsPerPoll = resolveMaxItemsPerPoll(context);

      return new PollingSourceWrapper<>(pollingSource, scheduler, maxItemsPerPoll, exceptionHandler, componentLocation);
    }
  }

  private SchedulingStrategy resolveSchedulingStrategy(ValueResolvingContext context) throws MuleException {
    ValueResolver<?> schedulingValueResolver = sourceParameters.getResolvers().get(SCHEDULING_STRATEGY_PARAMETER_NAME);
    if (schedulingValueResolver == null) {
      return new FixedFrequencyScheduler(60000, 0, MILLISECONDS);
    } else {
      return (SchedulingStrategy) schedulingValueResolver.resolve(context);
    }
  }

  private int resolveMaxItemsPerPoll(ValueResolvingContext context) throws MuleException {
    ValueResolver<?> valueResolver = sourceParameters.getResolvers().get(POLLING_SOURCE_LIMIT_PARAMETER_NAME);
    if (valueResolver == null) {
      return MAX_VALUE;
    } else {
      int maxItemsPerPoll = (Integer) valueResolver.resolve(context);
      if (maxItemsPerPoll < 1) {
        throw new IllegalArgumentException(format("The %s parameter must have a value greater than 1",
                                                  POLLING_SOURCE_LIMIT_PARAMETER_NAME));
      }
      return maxItemsPerPoll;
    }
  }

  private Optional<ConnectionProvider<?>> getConnectionProvider(Optional<ConfigurationInstance> configurationInstance,
                                                                SourceConnectionManager connectionManager,
                                                                ComponentLocation componentLocation)
      throws MuleException {
    if (!sourceModel.requiresConnection()) {
      return empty();
    }

    ConfigurationInstance config = configurationInstance.orElseThrow(() -> new DefaultMuleException(createStaticMessage(
                                                                                                                        "Message Source on root component '%s' requires a connection but it doesn't point to any configuration. Please review your "
                                                                                                                            + "application",
                                                                                                                        componentLocation
                                                                                                                            .getRootContainerName())));

    if (!config.getConnectionProvider().isPresent()) {
      throw new DefaultMuleException(createStaticMessage(format(
                                                                "Message Source on root component '%s' requires a connection, but points to config '%s' which doesn't specify any. "
                                                                    + "Please review your application",
                                                                componentLocation.getRootContainerName(),
                                                                config.getName())));
    }

    return of(new SourceConnectionProvider(connectionManager, config));
  }

  public ResolverSet getSourceParameters() {
    return sourceParameters;
  }
}
