/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.store;

import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.sameInstance;
import static org.hamcrest.MatcherAssert.assertThat;

import static org.junit.jupiter.api.Assertions.assertThrows;

import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;

import org.mule.runtime.api.store.ObjectAlreadyExistsException;
import org.mule.runtime.api.store.ObjectDoesNotExistException;
import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.runtime.api.store.ObjectStoreNotAvailableException;
import org.mule.sdk.api.store.ObjectStore;
import org.mule.tck.junit4.AbstractMuleTestCase;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class SdkObjectStoreAdapterTestCase extends AbstractMuleTestCase {

  private org.mule.runtime.api.store.ObjectStore muleObjectStore;

  @BeforeEach
  public void setup() {
    muleObjectStore = mock(org.mule.runtime.api.store.ObjectStore.class);
  }

  @Test
  void nullValueCannotBeAdaptedAndThrowsException() {
    assertThrows(NullPointerException.class, () -> SdkObjectStoreAdapter.from(null));
  }

  @Test
  void muleObjectStoreIsAdaptedToSdkObjectStore() {
    assertThat(SdkObjectStoreAdapter.from(muleObjectStore), instanceOf(ObjectStore.class));
  }

  @Test
  void sdkObjectStoreIsNotAdapted() {
    ObjectStore objectStore = SdkObjectStoreAdapter.from(muleObjectStore);
    assertThat(SdkObjectStoreAdapter.from(objectStore), sameInstance(objectStore));
  }

  @Test
  void muleObjectStoreExceptionIsThrown() throws Exception {
    doThrow(ObjectStoreException.class).when(muleObjectStore).clear();
    assertThrows(ObjectStoreException.class, () -> SdkObjectStoreAdapter.from(muleObjectStore).clear());
  }

  @Test
  void muleObjectAlreadyExistsExceptionIsThrown() throws Exception {
    doThrow(ObjectAlreadyExistsException.class).when(muleObjectStore).store("key", "value");
    assertThrows(ObjectAlreadyExistsException.class, () -> SdkObjectStoreAdapter.from(muleObjectStore).store("key", "value"));
  }

  @Test
  void muleObjectDoesNotExistExceptionIsThrown() throws Exception {
    doThrow(ObjectDoesNotExistException.class).when(muleObjectStore).remove("key");
    assertThrows(ObjectDoesNotExistException.class, () -> SdkObjectStoreAdapter.from(muleObjectStore).remove("key"));
  }

  @Test
  void muleObjectStoreNotAvailableExceptionIsThrown() throws Exception {
    doThrow(ObjectStoreNotAvailableException.class).when(muleObjectStore).contains("key");
    assertThrows(ObjectStoreNotAvailableException.class, () -> SdkObjectStoreAdapter.from(muleObjectStore).contains("key"));
  }
}
