/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.metadata;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.api.metadata.MetadataKeyBuilder.newKey;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.OutputModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.source.SourceCallbackModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataContext;
import org.mule.runtime.api.metadata.MetadataKey;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.descriptor.InputMetadataDescriptor;
import org.mule.runtime.api.metadata.descriptor.OutputMetadataDescriptor;
import org.mule.runtime.api.metadata.descriptor.RouterInputMetadataDescriptor;
import org.mule.runtime.api.metadata.descriptor.ScopeInputMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.extension.api.property.MetadataKeyIdModelProperty;
import org.mule.runtime.extension.api.property.MetadataKeyPartModelProperty;
import org.mule.runtime.module.extension.api.runtime.resolver.ParameterValueResolver;
import org.mule.runtime.module.extension.internal.util.ReflectionCache;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.mockito.junit.jupiter.MockitoSettings;
import org.mockito.quality.Strictness;

/**
 * Unit tests for {@link DefaultMetadataMediator}.
 *
 * This test class verifies the core functionality of the metadata mediation layer, including metadata key resolution,
 * input/output metadata resolution, and specialized resolution for scopes and routers.
 */
@SmallTest
@ExtendWith(MockitoExtension.class)
@MockitoSettings(strictness = Strictness.LENIENT)
class DefaultMetadataMediatorTestCase extends AbstractMuleTestCase {

  private static final String OPERATION_NAME = "testOperation";
  private static final String PARAM_NAME = "metadataKeyParam";
  private static final String KEY_ID = "testKey";
  private static final String KEY_CONTAINER_NAME = "keyContainer";

  private DefaultMetadataMediator<OperationModel> mediator;

  @Mock
  private OperationModel operationModel;

  @Mock
  private ConstructModel constructModel;

  @Mock
  private SourceModel sourceModel;

  @Mock
  private SourceCallbackModel successCallbackModel;

  @Mock
  private SourceCallbackModel errorCallbackModel;

  @Mock
  private MetadataContext metadataContext;

  @Mock
  private ParameterValueResolver parameterValueResolver;

  @Mock
  private ParameterModel parameterModel;

  @Mock
  private ParameterGroupModel parameterGroupModel;

  @Mock
  private OutputModel outputModel;

  @Mock
  private ReflectionCache reflectionCache;

  private MetadataType stringType;
  private List<ParameterModel> parameterModels;
  private List<ParameterGroupModel> parameterGroupModels;

  @BeforeEach
  void setUp() {
    stringType = BaseTypeBuilder.create(MetadataFormat.JAVA).stringType().build();

    // Setup parameter models
    parameterModels = new ArrayList<>();
    parameterModels.add(parameterModel);

    parameterGroupModels = new ArrayList<>();
    parameterGroupModels.add(parameterGroupModel);

    // Setup basic parameter model
    when(parameterModel.getName()).thenReturn(PARAM_NAME);
    when(parameterModel.getType()).thenReturn(stringType);
    when(parameterModel.isRequired()).thenReturn(true);
    when(parameterModel.getModelProperty(MetadataKeyPartModelProperty.class)).thenReturn(Optional.empty());

    // Setup parameter group model
    when(parameterGroupModel.getName()).thenReturn("General");
    when(parameterGroupModel.getParameterModels()).thenReturn(parameterModels);

    // Setup operation model
    when(operationModel.getName()).thenReturn(OPERATION_NAME);
    when(operationModel.getAllParameterModels()).thenReturn(parameterModels);
    when(operationModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(operationModel.getModelProperty(MetadataKeyIdModelProperty.class)).thenReturn(Optional.empty());

    // Setup output model
    when(outputModel.getType()).thenReturn(stringType);
    when(operationModel.getOutput()).thenReturn(outputModel);
    when(operationModel.getOutputAttributes()).thenReturn(outputModel);

    reflectionCache = new ReflectionCache();
  }

  @Test
  void createMediatorWithOperationModel() {
    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    assertThat(mediator, is(notNullValue()));
  }

  @Test
  void createMediatorWithConstructModel() {
    when(constructModel.getName()).thenReturn("testConstruct");
    when(constructModel.getAllParameterModels()).thenReturn(parameterModels);
    when(constructModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(constructModel.getModelProperty(MetadataKeyIdModelProperty.class)).thenReturn(Optional.empty());

    DefaultMetadataMediator<ConstructModel> constructMediator =
        new DefaultMetadataMediator<>(constructModel, reflectionCache);

    assertThat(constructMediator, is(notNullValue()));
  }

  @Test
  void createMediatorWithSourceModel() {
    when(sourceModel.getName()).thenReturn("testSource");
    when(sourceModel.getAllParameterModels()).thenReturn(parameterModels);
    when(sourceModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(sourceModel.getModelProperty(MetadataKeyIdModelProperty.class)).thenReturn(Optional.empty());
    when(sourceModel.getOutput()).thenReturn(outputModel);
    when(sourceModel.getOutputAttributes()).thenReturn(outputModel);
    when(sourceModel.getSuccessCallback()).thenReturn(Optional.of(successCallbackModel));
    when(sourceModel.getErrorCallback()).thenReturn(Optional.of(errorCallbackModel));

    // Setup callback models
    when(successCallbackModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(successCallbackModel.getAllParameterModels()).thenReturn(parameterModels);
    when(errorCallbackModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(errorCallbackModel.getAllParameterModels()).thenReturn(parameterModels);

    DefaultMetadataMediator<SourceModel> sourceMediator =
        new DefaultMetadataMediator<>(sourceModel, reflectionCache);

    assertThat(sourceMediator, is(notNullValue()));
  }

  @Test
  void getMetadataKeysWithoutMetadataKeyId() {
    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    MetadataResult<MetadataKeysContainer> result = mediator.getMetadataKeys(metadataContext);

    assertThat(result, is(notNullValue()));
    assertThat(result.isSuccess(), is(true));
  }

  @Test
  @SuppressWarnings("deprecation")
  void getMetadataKeysWithMetadataKeyIdModelProperty() {
    MetadataKeyIdModelProperty keyIdModelProperty =
        new MetadataKeyIdModelProperty(stringType, KEY_CONTAINER_NAME);
    when(operationModel.getModelProperty(MetadataKeyIdModelProperty.class))
        .thenReturn(Optional.of(keyIdModelProperty));

    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    MetadataResult<MetadataKeysContainer> result = mediator.getMetadataKeys(metadataContext);

    assertThat(result, is(notNullValue()));
    assertThat(result.isSuccess(), is(true));
    assertThat(result.get(), is(notNullValue()));
  }

  @Test
  @SuppressWarnings("deprecation")
  void getMetadataKeysWithParameterValueResolver() throws Exception {
    MetadataKeyIdModelProperty keyIdModelProperty =
        new MetadataKeyIdModelProperty(stringType, KEY_CONTAINER_NAME);
    when(operationModel.getModelProperty(MetadataKeyIdModelProperty.class))
        .thenReturn(Optional.of(keyIdModelProperty));
    when(parameterValueResolver.getParameterValue(KEY_CONTAINER_NAME)).thenReturn(KEY_ID);

    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    MetadataResult<MetadataKeysContainer> result =
        mediator.getMetadataKeys(metadataContext, parameterValueResolver);

    assertThat(result, is(notNullValue()));
    assertThat(result.isSuccess(), is(true));
    assertThat(result.get(), is(notNullValue()));
  }

  @Test
  void getMetadataKeysWithPartialKey() {
    MetadataKey partialKey = newKey(KEY_ID).build();

    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    MetadataResult<MetadataKeysContainer> result =
        mediator.getMetadataKeys(metadataContext, partialKey);

    assertThat(result, is(notNullValue()));
    assertThat(result.isSuccess(), is(true));
    assertThat(result.get(), is(notNullValue()));
  }

  @Test
  void getMetadataWithMetadataKey() {
    MetadataKey key = newKey(KEY_ID).build();

    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    MetadataResult<ComponentMetadataDescriptor<OperationModel>> result =
        mediator.getMetadata(metadataContext, key);

    assertThat(result, is(notNullValue()));
    assertThat(result.isSuccess(), is(true));
    ComponentMetadataDescriptor<OperationModel> descriptor = result.get();
    assertThat(descriptor, is(notNullValue()));
  }

  @Test
  void getMetadataWithParameterValueResolver() throws Exception {
    when(parameterValueResolver.getParameterValue(any())).thenReturn(null);

    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    MetadataResult<ComponentMetadataDescriptor<OperationModel>> result =
        mediator.getMetadata(metadataContext, parameterValueResolver);

    assertThat(result, is(notNullValue()));
    assertThat(result.isSuccess(), is(true));
    ComponentMetadataDescriptor<OperationModel> descriptor = result.get();
    assertThat(descriptor, is(notNullValue()));
  }

  @Test
  void getInputMetadata() {
    MetadataKey key = newKey(KEY_ID).build();

    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    MetadataResult<InputMetadataDescriptor> result =
        mediator.getInputMetadata(metadataContext, key);

    assertThat(result, is(notNullValue()));
    assertThat(result.isSuccess(), is(true));
    InputMetadataDescriptor descriptor = result.get();
    assertThat(descriptor, is(notNullValue()));
    assertThat(descriptor.getAllParameters().isEmpty(), is(false));
  }

  @Test
  void getOutputMetadata() {
    MetadataKey key = newKey(KEY_ID).build();

    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    MetadataResult<OutputMetadataDescriptor> result =
        mediator.getOutputMetadata(metadataContext, key);

    assertThat(result, is(notNullValue()));
    assertThat(result.isSuccess(), is(true));
    OutputMetadataDescriptor descriptor = result.get();
    assertThat(descriptor, is(notNullValue()));
    assertThat(descriptor.getPayloadMetadata(), is(notNullValue()));
    assertThat(descriptor.getAttributesMetadata(), is(notNullValue()));
  }

  @Test
  void getOutputMetadataFailsWhenComponentHasNoOutput() {
    ComponentModel componentWithoutOutput = mock(ComponentModel.class);
    when(componentWithoutOutput.getName()).thenReturn("noOutputComponent");
    when(componentWithoutOutput.getAllParameterModels()).thenReturn(parameterModels);
    when(componentWithoutOutput.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(componentWithoutOutput.getModelProperty(MetadataKeyIdModelProperty.class))
        .thenReturn(Optional.empty());

    DefaultMetadataMediator<ComponentModel> componentMediator =
        new DefaultMetadataMediator<>(componentWithoutOutput, reflectionCache);

    MetadataKey key = newKey(KEY_ID).build();
    MetadataResult<OutputMetadataDescriptor> result =
        componentMediator.getOutputMetadata(metadataContext, key);

    assertThat(result, is(notNullValue()));
    assertThat(result.isSuccess(), is(false));
    assertThat(result.getFailures().isEmpty(), is(false));
    assertThat(result.getFailures().size(), is(1));
  }

  @Test
  void getScopeInputMetadataFailsWhenComponentIsNotScope() {
    MetadataKey key = newKey(KEY_ID).build();

    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    MetadataResult<ScopeInputMetadataDescriptor> result =
        mediator.getScopeInputMetadata(metadataContext, key, () -> null, null);

    assertThat(result, is(notNullValue()));
    assertThat(result.isSuccess(), is(false));
    assertThat(result.getFailures().isEmpty(), is(false));
    assertThat(result.getFailures().size(), is(1));
  }

  @Test
  void getRouterInputMetadataFailsWhenComponentIsNotRouter() {
    MetadataKey key = newKey(KEY_ID).build();

    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    MetadataResult<RouterInputMetadataDescriptor> result =
        mediator.getRouterInputMetadata(metadataContext, key, () -> null, null);

    assertThat(result, is(notNullValue()));
    assertThat(result.isSuccess(), is(false));
    assertThat(result.getFailures().isEmpty(), is(false));
    assertThat(result.getFailures().size(), is(1));
  }

  @Test
  void mediatorHandlesMetadataKeyPartModelProperty() {
    // Setup parameter with MetadataKeyPartModelProperty
    when(parameterModel.getModelProperty(MetadataKeyPartModelProperty.class))
        .thenReturn(Optional.of(new MetadataKeyPartModelProperty(1)));

    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    assertThat(mediator, is(notNullValue()));
  }

  @Test
  void mediatorHandlesMultipleParametersWithKeyParts() {
    // Setup multiple parameters with MetadataKeyPartModelProperty
    ParameterModel param1 = mock(ParameterModel.class);
    ParameterModel param2 = mock(ParameterModel.class);

    when(param1.getName()).thenReturn("keyPart1");
    when(param1.getType()).thenReturn(stringType);
    when(param1.getModelProperty(MetadataKeyPartModelProperty.class))
        .thenReturn(Optional.of(new MetadataKeyPartModelProperty(1)));

    when(param2.getName()).thenReturn("keyPart2");
    when(param2.getType()).thenReturn(stringType);
    when(param2.getModelProperty(MetadataKeyPartModelProperty.class))
        .thenReturn(Optional.of(new MetadataKeyPartModelProperty(2)));

    List<ParameterModel> multipleParams = new ArrayList<>();
    multipleParams.add(param1);
    multipleParams.add(param2);

    when(operationModel.getAllParameterModels()).thenReturn(multipleParams);

    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    assertThat(mediator, is(notNullValue()));
  }

  @Test
  @SuppressWarnings("deprecation")
  void getMetadataWithNullKeyWhenKeyIsRequired() throws Exception {
    MetadataKeyIdModelProperty keyIdModelProperty =
        new MetadataKeyIdModelProperty(stringType, KEY_CONTAINER_NAME);
    when(operationModel.getModelProperty(MetadataKeyIdModelProperty.class))
        .thenReturn(Optional.of(keyIdModelProperty));

    // Setup parameter as required metadata key
    when(parameterModel.getModelProperty(MetadataKeyPartModelProperty.class))
        .thenReturn(Optional.of(new MetadataKeyPartModelProperty(1)));
    when(parameterModel.isRequired()).thenReturn(true);
    when(parameterValueResolver.getParameterValue(any())).thenReturn(null);

    mediator = new DefaultMetadataMediator<>(operationModel, reflectionCache);

    MetadataResult<ComponentMetadataDescriptor<OperationModel>> result =
        mediator.getMetadata(metadataContext, parameterValueResolver);

    assertThat(result, is(notNullValue()));
    // When metadata key is required but null, the result may fail validation
    // so we just verify the result exists
  }

  @Test
  void getInputMetadataWithSourceCallbacks() {
    // Setup source model with callbacks
    when(sourceModel.getName()).thenReturn("testSource");
    when(sourceModel.getAllParameterModels()).thenReturn(parameterModels);
    when(sourceModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(sourceModel.getModelProperty(MetadataKeyIdModelProperty.class)).thenReturn(Optional.empty());
    when(sourceModel.getOutput()).thenReturn(outputModel);
    when(sourceModel.getOutputAttributes()).thenReturn(outputModel);
    when(sourceModel.getSuccessCallback()).thenReturn(Optional.of(successCallbackModel));
    when(sourceModel.getErrorCallback()).thenReturn(Optional.of(errorCallbackModel));

    // Setup callback models
    when(successCallbackModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(successCallbackModel.getAllParameterModels()).thenReturn(parameterModels);
    when(errorCallbackModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(errorCallbackModel.getAllParameterModels()).thenReturn(parameterModels);

    DefaultMetadataMediator<SourceModel> sourceMediator =
        new DefaultMetadataMediator<>(sourceModel, reflectionCache);

    MetadataKey key = newKey(KEY_ID).build();
    MetadataResult<InputMetadataDescriptor> result =
        sourceMediator.getInputMetadata(metadataContext, key);

    assertThat(result, is(notNullValue()));
    assertThat(result.isSuccess(), is(true));
    InputMetadataDescriptor descriptor = result.get();
    assertThat(descriptor, is(notNullValue()));
    assertThat(descriptor.getAllParameters().isEmpty(), is(false));
  }
}

