/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.source;

import static org.mule.runtime.module.extension.api.runtime.resolver.ParameterValueResolver.staticParametersFrom;
import static org.mule.runtime.module.extension.api.util.MuleExtensionUtils.loadExtension;

import static java.util.Map.of;

import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.parameterization.ComponentParameterization;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.Extension;
import org.mule.runtime.extension.api.annotation.Sources;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.runtime.source.Source;
import org.mule.runtime.extension.api.runtime.source.SourceCallback;
import org.mule.runtime.extension.api.runtime.source.SourceFactoryContext;
import org.mule.runtime.module.extension.api.runtime.resolver.ParameterValueResolver;
import org.mule.runtime.module.extension.internal.loader.parser.java.source.JavaSdkSourceFactory;
import org.mule.runtime.module.extension.internal.runtime.ResolverBasedComponentParameterization;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;
import org.mule.test.heisenberg.extension.HeisenbergSource;

import java.lang.reflect.InvocationTargetException;

import org.hamcrest.Matchers;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

@SmallTest
public class JavaSdkSourceFactoryTestCase extends AbstractMuleTestCase {

  @Rule
  public ExpectedException expectedException = ExpectedException.none();

  @Test
  public void create() {
    assertThat(new JavaSdkSourceFactory(HeisenbergSource.class).createMessageSource().getValue().get(),
               is(instanceOf(HeisenbergSource.class)));
  }

  @Test
  public void nullType() {
    expectedException.expect(IllegalArgumentException.class);
    new JavaSdkSourceFactory(null);
  }

  @Test
  public void notInstantiable() {
    expectedException.expect(IllegalArgumentException.class);
    new JavaSdkSourceFactory(Source.class);
  }

  @Test
  public void exceptionOnInstantiation() {
    expectedException.expect(RuntimeException.class);
    expectedException.expectCause(Matchers.instanceOf(InvocationTargetException.class));
    new JavaSdkSourceFactory(UncreatableSource.class).createMessageSource();
  }

  @Test
  public void sourceWithParameterGroupWithNonParameterField() {
    ExtensionModel extensionModel = loadExtension(TestExtension.class);
    ParameterValueResolver parameterValueResolver = staticParametersFrom(of("someField", "someValue"));
    ComponentParameterization parameterization = new ResolverBasedComponentParameterization<>(extensionModel
        .getSourceModel("SourceWithParameterGroupWithNonParameterField").get(), parameterValueResolver);
    SourceFactoryContext context = mock(SourceFactoryContext.class);

    when(context.getParameterization()).thenReturn(parameterization);
    assertThat(new JavaSdkSourceFactory(SourceWithParameterGroupWithNonParameterField.class).createMessageSource(context),
               is(instanceOf(SourceWithParameterGroupWithNonParameterField.class)));
  }

  public static class UncreatableSource extends Source {

    public UncreatableSource() {
      throw new IllegalArgumentException();
    }

    @Override
    public void onStart(SourceCallback sourceCallback) throws MuleException {

    }

    @Override
    public void onStop() {

    }
  }

  @Extension(name = "Test extension")
  @Sources(SourceWithParameterGroupWithNonParameterField.class)
  public static class TestExtension {

  }

  @Alias("SourceWithParameterGroupWithNonParameterField")
  public static class SourceWithParameterGroupWithNonParameterField extends Source<String, Object> {

    @ParameterGroup(name = "withNonParameterField")
    private ParameterGroupWithNonParameterField parameterGroupWithNonParameterField;

    @Override
    public void onStart(SourceCallback sourceCallback) throws MuleException {

    }

    @Override
    public void onStop() {

    }
  }

  public static class ParameterGroupWithNonParameterField {

    public static final String SOME_MESSAGE = "Some message";

    @Parameter
    private String someField;

  }

}
