/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.store;

import static org.mule.test.allure.AllureConstants.ObjectStoreFeature.OS_SUPPORT;
import static org.mule.test.allure.AllureConstants.ObjectStoreFeature.ObjectStoreStory.LOCAL_OS_CONFIG;

import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.sameInstance;

import static org.junit.jupiter.api.Assertions.assertThrows;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.sdk.api.store.ObjectStore;
import org.mule.sdk.api.store.ObjectStoreManager;
import org.mule.sdk.api.store.ObjectStoreSettings;
import org.mule.tck.junit4.AbstractMuleTestCase;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentCaptor;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(OS_SUPPORT)
class SdkObjectStoreManagerAdapterTestCase extends AbstractMuleTestCase {

  private static final String OS_NAME = "test-os";
  private static final String LOCAL_OS_NAME = "local-test-os";

  private org.mule.runtime.api.store.ObjectStoreManager muleObjectStoreManager;
  private org.mule.runtime.api.store.ObjectStoreManager localMuleObjectStoreManager;
  private org.mule.runtime.api.store.ObjectStore muleObjectStore;
  private org.mule.runtime.api.store.ObjectStore localMuleObjectStore;
  private SdkObjectStoreManagerAdapter sdkObjectStoreManagerAdapter;

  @BeforeEach
  public void setup() {
    muleObjectStoreManager = mock(org.mule.runtime.api.store.ObjectStoreManager.class);
    muleObjectStore = mock(org.mule.runtime.api.store.ObjectStore.class);
    when(muleObjectStoreManager.createObjectStore(eq(OS_NAME), any())).thenReturn(muleObjectStore);
    when(muleObjectStoreManager.getObjectStore(eq(OS_NAME))).thenReturn(muleObjectStore);
    when(muleObjectStoreManager.getOrCreateObjectStore(eq(OS_NAME), any())).thenReturn(muleObjectStore);

    localMuleObjectStoreManager = mock(org.mule.runtime.api.store.ObjectStoreManager.class);
    localMuleObjectStore = mock(org.mule.runtime.api.store.ObjectStore.class);
    when(localMuleObjectStoreManager.createObjectStore(eq(LOCAL_OS_NAME), any())).thenReturn(localMuleObjectStore);
    when(localMuleObjectStoreManager.getObjectStore(eq(LOCAL_OS_NAME))).thenReturn(localMuleObjectStore);
    when(localMuleObjectStoreManager.getOrCreateObjectStore(eq(LOCAL_OS_NAME), any())).thenReturn(localMuleObjectStore);

    sdkObjectStoreManagerAdapter = new SdkObjectStoreManagerAdapter();
    sdkObjectStoreManagerAdapter.setDelegate(muleObjectStoreManager);
    sdkObjectStoreManagerAdapter.setLocalDelegate(localMuleObjectStoreManager);
  }

  @Test
  void adapterIsAnSdkObjectStoreManager() {
    assertThat(sdkObjectStoreManagerAdapter, instanceOf(ObjectStoreManager.class));
  }

  @Test
  void muleObjectStoreIsCreatedAndAdaptedToSdkObjectStore() {
    ObjectStore sdkObjectStore = sdkObjectStoreManagerAdapter.createObjectStore(OS_NAME, ObjectStoreSettings.builder().build());
    assertThat(sdkObjectStore, instanceOf(SdkObjectStoreAdapter.class));
    verify(muleObjectStoreManager, times(1)).createObjectStore(eq(OS_NAME), any());
  }

  @Test
  void muleObjectStoreIsRetrievedAndAdaptedToSdkObjectStore() {
    ObjectStore sdkObjectStore = sdkObjectStoreManagerAdapter.getObjectStore(OS_NAME);
    assertThat(sdkObjectStore, instanceOf(SdkObjectStoreAdapter.class));
    verify(muleObjectStoreManager, times(1)).getObjectStore(eq(OS_NAME));
  }

  @Test
  void muleObjectStoreIsCreatedOrRetrievedAndAdaptedToSdkObjectStore() {
    ObjectStore sdkObjectStore =
        sdkObjectStoreManagerAdapter.getOrCreateObjectStore(OS_NAME, ObjectStoreSettings.builder().build());
    assertThat(sdkObjectStore, instanceOf(SdkObjectStoreAdapter.class));
    verify(muleObjectStoreManager, times(1)).getOrCreateObjectStore(eq(OS_NAME), any());
  }

  @Test
  void muleObjectStoreIsDisposed() throws Exception {
    sdkObjectStoreManagerAdapter.disposeStore(OS_NAME);
    verify(muleObjectStoreManager, times(1)).disposeStore(eq(OS_NAME));
  }

  @Test
  void sdkObjectStoreSettingsAreConvertedToMuleObjectStoreSettings() {
    ObjectStoreSettings sdkObjectStoreSettings = ObjectStoreSettings.builder()
        .persistent(true)
        .maxEntries(100)
        .entryTtl(1000L)
        .expirationInterval(5000L)
        .build();
    sdkObjectStoreManagerAdapter.createObjectStore(OS_NAME, sdkObjectStoreSettings);

    ArgumentCaptor<org.mule.runtime.api.store.ObjectStoreSettings> settingsArgumentCaptor = ArgumentCaptor.forClass(
                                                                                                                    org.mule.runtime.api.store.ObjectStoreSettings.class);
    verify(muleObjectStoreManager, times(1)).createObjectStore(eq(OS_NAME), settingsArgumentCaptor.capture());
    org.mule.runtime.api.store.ObjectStoreSettings muleObjectStoreSettings = settingsArgumentCaptor.getValue();

    assertThat(muleObjectStoreSettings.isPersistent(), is(true));
    assertThat(muleObjectStoreSettings.getMaxEntries().get(), is(100));
    assertThat(muleObjectStoreSettings.getEntryTTL().get(), is(1000L));
    assertThat(muleObjectStoreSettings.getExpirationInterval(), is(5000L));
  }

  @Test
  void defaultSdkObjectStoreSettingsAreConvertedToDefaultMuleObjectStoreSettings() {
    sdkObjectStoreManagerAdapter.createObjectStore(OS_NAME, ObjectStoreSettings.builder().build());

    ArgumentCaptor<org.mule.runtime.api.store.ObjectStoreSettings> settingsArgumentCaptor = ArgumentCaptor.forClass(
                                                                                                                    org.mule.runtime.api.store.ObjectStoreSettings.class);
    verify(muleObjectStoreManager, times(1)).createObjectStore(eq(OS_NAME), settingsArgumentCaptor.capture());
    org.mule.runtime.api.store.ObjectStoreSettings muleObjectStoreSettings = settingsArgumentCaptor.getValue();

    assertThat(muleObjectStoreSettings.isPersistent(), is(true));
    assertThat(muleObjectStoreSettings.getMaxEntries().isPresent(), is(false));
    assertThat(muleObjectStoreSettings.getEntryTTL().isPresent(), is(false));
    assertThat(muleObjectStoreSettings.getExpirationInterval(),
               is(org.mule.runtime.api.store.ObjectStoreSettings.DEFAULT_EXPIRATION_INTERVAL));
  }

  @Test
  void onFailureMuleObjectStoreExceptionIsThrown() throws Exception {
    doThrow(ObjectStoreException.class).when(muleObjectStoreManager).disposeStore(eq(OS_NAME));
    assertThrows(ObjectStoreException.class, () -> sdkObjectStoreManagerAdapter.disposeStore(OS_NAME));
  }

  @Story(LOCAL_OS_CONFIG)
  @Test
  void localOSManagerLifecycle() throws ObjectStoreException {
    var localOS = sdkObjectStoreManagerAdapter.createObjectStore(LOCAL_OS_NAME,
                                                                 ObjectStoreSettings.builder().distributed(false).build());

    localOS.clear();
    verify(muleObjectStore, never()).clear();

    sdkObjectStoreManagerAdapter.disposeStore(LOCAL_OS_NAME);

    verify(muleObjectStoreManager, never()).disposeStore(LOCAL_OS_NAME);
    verify(localMuleObjectStoreManager, times(1)).disposeStore(LOCAL_OS_NAME);
  }

  @Story(LOCAL_OS_CONFIG)
  @Test
  void localOSManagerGetOrCreateLifecycle() throws ObjectStoreException {
    var localOS = sdkObjectStoreManagerAdapter.getOrCreateObjectStore(LOCAL_OS_NAME,
                                                                      ObjectStoreSettings.builder().distributed(false).build());

    assertThat(((SdkObjectStoreAdapter) sdkObjectStoreManagerAdapter.getObjectStore(LOCAL_OS_NAME)).getDelegate(),
               sameInstance(((SdkObjectStoreAdapter) localOS).getDelegate()));

    verify(muleObjectStoreManager, never()).getObjectStore(LOCAL_OS_NAME);
    verify(localMuleObjectStoreManager, times(1)).getObjectStore(LOCAL_OS_NAME);
  }

  @Story(LOCAL_OS_CONFIG)
  @Test
  void distributedOSManagerLifecycle() throws ObjectStoreException {
    var distributedOS = sdkObjectStoreManagerAdapter.createObjectStore(OS_NAME,
                                                                       ObjectStoreSettings.builder().distributed(true).build());

    distributedOS.clear();
    verify(muleObjectStore, times(1)).clear();

    sdkObjectStoreManagerAdapter.disposeStore(OS_NAME);

    verify(muleObjectStoreManager, times(1)).disposeStore(OS_NAME);
    verify(localMuleObjectStoreManager, never()).disposeStore(OS_NAME);
  }
}
