/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.api.loader.java.type;

import static org.mule.runtime.module.extension.internal.loader.parser.java.MuleExtensionAnnotationParser.mapReduceAnnotation;
import static org.mule.runtime.module.extension.internal.util.MuleExtensionUtils.getDefaultValue;

import static java.lang.String.format;

import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.retry.policy.RetryPolicyTemplate;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.DefaultEncoding;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.client.ExtensionsClient;
import org.mule.runtime.extension.api.exception.IllegalParameterModelDefinitionException;
import org.mule.runtime.extension.api.notification.NotificationEmitter;
import org.mule.runtime.extension.api.runtime.operation.FlowListener;
import org.mule.runtime.extension.api.runtime.parameter.CorrelationInfo;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.extension.api.runtime.process.VoidCompletionCallback;
import org.mule.runtime.extension.api.runtime.route.Chain;
import org.mule.runtime.extension.api.runtime.source.SourceCallbackContext;
import org.mule.runtime.extension.api.runtime.source.SourceCompletionCallback;
import org.mule.runtime.extension.api.runtime.source.SourceResult;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.extension.api.security.AuthenticationHandler;
import org.mule.sdk.api.connectivity.oauth.AuthCodeRequest;
import org.mule.sdk.api.runtime.parameter.Literal;
import org.mule.sdk.api.runtime.parameter.ParameterResolver;
import org.mule.sdk.api.runtime.source.DistributedTraceContextManager;

import java.lang.reflect.AnnotatedElement;
import java.util.Set;

import javax.lang.model.element.VariableElement;

/**
 * A contract for any kind of component from which an extension's parameter can be derived
 *
 * @since 4.0
 */
@NoImplement
public interface ExtensionParameter extends WithType, WithAnnotations, NamedObject, WithAlias, WithOwner, WithElement {

  Set<Class<?>> IMPLICIT_ARGUMENT_TYPES = Set.of(Error.class,
                                                 SourceCallbackContext.class,
                                                 org.mule.sdk.api.runtime.source.SourceCallbackContext.class,
                                                 CompletionCallback.class,
                                                 org.mule.sdk.api.runtime.process.CompletionCallback.class,
                                                 VoidCompletionCallback.class,
                                                 org.mule.sdk.api.runtime.process.VoidCompletionCallback.class,
                                                 SourceCompletionCallback.class,
                                                 org.mule.sdk.api.runtime.source.SourceCompletionCallback.class,
                                                 MediaType.class,
                                                 AuthenticationHandler.class,
                                                 org.mule.sdk.api.security.AuthenticationHandler.class,
                                                 FlowListener.class,
                                                 org.mule.sdk.api.runtime.operation.FlowListener.class,
                                                 StreamingHelper.class,
                                                 org.mule.sdk.api.runtime.streaming.StreamingHelper.class,
                                                 SourceResult.class,
                                                 org.mule.sdk.api.runtime.source.SourceResult.class,
                                                 ComponentLocation.class,
                                                 Chain.class,
                                                 org.mule.sdk.api.runtime.route.Chain.class,
                                                 CorrelationInfo.class,
                                                 DistributedTraceContextManager.class,
                                                 org.mule.sdk.api.runtime.parameter.CorrelationInfo.class,
                                                 NotificationEmitter.class,
                                                 org.mule.sdk.api.notification.NotificationEmitter.class,
                                                 ExtensionsClient.class,
                                                 org.mule.sdk.api.client.ExtensionsClient.class,
                                                 RetryPolicyTemplate.class);

  Set<String> IMPLICIT_ARGUMENT_PACKAGES = Set.of("org.mule.sdk.api");

  Set<Class<?>> EXPLICIT_MULE_ARGUMENT_TYPES = Set.of(TypedValue.class,
                                                      ParameterResolver.class,
                                                      org.mule.runtime.extension.api.runtime.parameter.ParameterResolver.class,
                                                      org.mule.runtime.extension.api.runtime.parameter.Literal.class,
                                                      Literal.class,
                                                      TlsContextFactory.class,
                                                      ObjectStore.class,
                                                      AuthCodeRequest.class);

  /**
   * @return A {@code boolean} indicating whether the parameter should be advertised and added as a {@link ParameterModel} in the
   *         {@link ExtensionModel}
   */
  default boolean shouldBeAdvertised() {
    return !(IMPLICIT_ARGUMENT_TYPES.stream().anyMatch(aClass -> getType().isAssignableTo(aClass))
        || isAnnotatedWith(Config.class) || isAnnotatedWith(org.mule.sdk.api.annotation.param.Config.class)
        || isAnnotatedWith(org.mule.sdk.api.annotation.param.Connection.class) || isAnnotatedWith(Connection.class)
        || isAnnotatedWith(DefaultEncoding.class) || isAnnotatedWith(org.mule.sdk.api.annotation.param.DefaultEncoding.class))
        && (!IMPLICIT_ARGUMENT_PACKAGES.stream().anyMatch(packageName -> getType().getTypeName().startsWith(packageName))
            || EXPLICIT_MULE_ARGUMENT_TYPES.stream().anyMatch(aClass -> getType().isAssignableTo(aClass)));
  }

  /**
   * @return A {@code boolean} indicating whether the parameter is a required or not
   */
  default boolean isRequired() {
    return !(isAnnotatedWith(Optional.class) || isAnnotatedWith(org.mule.sdk.api.annotation.param.Optional.class));
  }

  /**
   * @return The {@link java.util.Optional} default value of the operation
   */
  default java.util.Optional<String> defaultValue() {
    return mapReduceAnnotation(this, Optional.class, org.mule.sdk.api.annotation.param.Optional.class,
                               legacyAnnotationValueFetcher -> getDefaultValue(legacyAnnotationValueFetcher
                                   .getStringValue(Optional::defaultValue)),
                               sdkAnnotationValueFetcher -> getDefaultValue(sdkAnnotationValueFetcher
                                   .getStringValue(org.mule.sdk.api.annotation.param.Optional::defaultValue)),
                               () -> new IllegalParameterModelDefinitionException(format("Parameter '%s' is annotated with '@%s' and '@%s' at the same time",
                                                                                         getName(),
                                                                                         Optional.class.getName(),
                                                                                         org.mule.sdk.api.annotation.param.Optional.class
                                                                                             .getName())));
  }

  /**
   * @return The {@link AnnotatedElement} from which {@code this} instance was derived
   */
  java.util.Optional<? extends AnnotatedElement> getDeclaringElement();

  @Override
  java.util.Optional<VariableElement> getElement();
}
