/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime;

import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.getMemberName;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.parameterization.ComponentParameterization;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.module.extension.api.runtime.resolver.ParameterValueResolver;
import org.mule.runtime.module.extension.api.runtime.resolver.ValueResolvingException;
import org.mule.runtime.module.extension.internal.runtime.config.ValueResolverNotFoundException;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * An implementation of {@link ComponentParameterization} which delegates to a {@link ParameterValueResolver}.
 *
 * @since 4.10.0
 */
public class ResolverBasedComponentParameterization<M extends ParameterizedModel> implements ComponentParameterization<M> {

  private final M model;
  private final ParameterValueResolver resolver;

  public ResolverBasedComponentParameterization(M model, ParameterValueResolver resolver) {
    this.model = model;
    this.resolver = resolver;
  }

  @Override
  public M getModel() {
    return model;
  }

  @Override
  public Object getParameter(String parameterGroupName, String parameterName) {
    try {
      // FIXME W-19057747: this is ignoring the group
      return getParameterValue(parameterName);
    } catch (ValueResolvingException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public Object getParameter(ParameterGroupModel parameterGroupModel, ParameterModel parameterModel) {
    try {
      // FIXME W-19057747: this is ignoring the group
      return getParameterValue(parameterModel);
    } catch (ValueResolvingException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public Map<Pair<ParameterGroupModel, ParameterModel>, Object> getParameters() {
    Map<Pair<ParameterGroupModel, ParameterModel>, Object> parameterValues = new HashMap<>();
    forEachParameter((pgm, pm, value) -> parameterValues.put(new Pair<>(pgm, pm), value));
    return parameterValues;
  }

  @Override
  public void forEachParameter(ParameterAction parameterAction) {
    for (ParameterGroupModel pgm : model.getParameterGroupModels()) {
      for (ParameterModel pm : pgm.getParameterModels()) {
        try {
          // FIXME W-19057747: this is ignoring the group
          Object value = getParameterValue(pm);
          if (value != null) {
            parameterAction.accept(pgm, pm, value);
          }
        } catch (ValueResolvingException e) {
          throw new RuntimeException(e);
        }
      }
    }
  }

  @Override
  public Optional<ComponentIdentifier> getComponentIdentifier() {
    // TODO W-19057747: implement
    return Optional.empty();
  }

  private Object getParameterValue(ParameterModel parameterModel) throws ValueResolvingException {
    // For some reason, parameters that are modeled with Fields in Java SDK are un-aliased in the ParameterValueResolver, this is
    // not the same that happens with parameters that are modeled from method arguments. See ParametersResolver#addToResolverSet.
    // This is the reason we need to use getMemberName here, but not getImplementingName.
    String memberName = getMemberName(parameterModel);
    try {
      return resolver.getParameterValue(memberName);
    } catch (ValueResolverNotFoundException e) {
      // Some implementations of ParameterValueResolver throw an exception if there is no resolver for the parameter
      // For ComponentParameterization it is valid to not have values for certain parameters, so we need to suppress it.
      return null;
    }
  }

  private Object getParameterValue(String parameterName) throws ValueResolvingException {
    return getParameterValue(model.getAllParameterModels().stream()
        .filter(pm -> pm.getName().equals(parameterName))
        .findFirst()
        .orElseThrow(() -> new ValueResolvingException("Parameter with name `" + parameterName + "` not found in model")));
  }
}
