/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.loader.parser.java.test;

import static org.mule.runtime.module.extension.internal.loader.parser.java.utils.MinMuleVersionUtils.declareWithBaselineMmv;
import static org.mule.runtime.module.extension.internal.loader.parser.java.utils.MinMuleVersionUtils.declarationWithMmv;
import static org.mule.runtime.module.extension.internal.loader.parser.java.utils.MinMuleVersionUtils.declarerWithMmv;
import static org.mule.runtime.module.extension.internal.loader.parser.java.utils.ResolvedMinMuleVersion.FIRST_MULE_VERSION;

import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.declaration.fluent.HasMinMuleVersionDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.OperationDeclarer;
import org.mule.runtime.api.meta.model.declaration.fluent.WithMinMuleVersionDeclaration;
import org.mule.runtime.extension.api.loader.ExtensionLoadingContext;
import org.mule.runtime.extension.api.loader.parser.MinMuleVersionParser;
import org.mule.runtime.module.extension.internal.loader.parser.java.utils.ResolvedMinMuleVersion;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class MinMuleVersionUtilsTestCase {

  private static final MuleVersion TEST_VERSION_4_6 = new MuleVersion("4.6");
  private static final MuleVersion TEST_VERSION_4_4 = new MuleVersion("4.4");
  private static final MuleVersion TEST_VERSION_4_8 = new MuleVersion("4.8");

  private ExtensionLoadingContext contextResolvingMinMuleVersion;
  private ExtensionLoadingContext contextNotResolvingMinMuleVersion;
  private HasMinMuleVersionDeclarer<?> mockDeclarer;
  private WithMinMuleVersionDeclaration mockDeclaration;
  private MinMuleVersionParser resolvedMMV;

  @BeforeEach
  void setUp() {
    contextResolvingMinMuleVersion = MinMuleVersionTestUtils.ctxResolvingMinMuleVersion();
    contextNotResolvingMinMuleVersion = mock(ExtensionLoadingContext.class);
    when(contextNotResolvingMinMuleVersion.isResolveMinMuleVersion()).thenReturn(false);
    mockDeclarer = mock(OperationDeclarer.class);
    mockDeclaration = mock(WithMinMuleVersionDeclaration.class);
    resolvedMMV = new ResolvedMinMuleVersion("testComponent", TEST_VERSION_4_6,
                                             "Test component has min mule version 4.6 for testing purposes.");
  }

  @Test
  void testDeclarerWithMmv() {
    declarerWithMmv(mockDeclarer, resolvedMMV);
    verify(mockDeclarer).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclarationWithMmv() {
    declarationWithMmv(mockDeclaration, resolvedMMV);
    verify(mockDeclaration).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarerWhenContextNotResolving() {
    declareWithBaselineMmv(() -> of(resolvedMMV), mockDeclarer, empty(), contextNotResolvingMinMuleVersion);
    verify(mockDeclarer, never()).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarerWhenContextResolving() {
    declareWithBaselineMmv(() -> of(resolvedMMV), mockDeclarer, empty(), contextResolvingMinMuleVersion);
    verify(mockDeclarer).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarerWhenNoResolvedMMV() {
    declareWithBaselineMmv(() -> empty(), mockDeclarer, empty(), contextResolvingMinMuleVersion);
    verify(mockDeclarer, never()).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarerWhenBaselineLower() {
    // Baseline (4.4) is lower than element MMV (4.6), so element MMV should be applied
    declareWithBaselineMmv(() -> of(resolvedMMV), mockDeclarer, of(TEST_VERSION_4_4),
                           contextResolvingMinMuleVersion);
    verify(mockDeclarer).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarerWhenBaselineHigher() {
    // Baseline (4.8) is higher than element MMV (4.6), so element MMV should NOT be applied
    declareWithBaselineMmv(() -> of(resolvedMMV), mockDeclarer, of(TEST_VERSION_4_8),
                           contextResolvingMinMuleVersion);
    verify(mockDeclarer, never()).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarerWhenBaselineEqual() {
    // Baseline (4.6) is equal to element MMV (4.6), baseline is not "newer than", so element MMV should be applied
    declareWithBaselineMmv(() -> of(resolvedMMV), mockDeclarer, of(TEST_VERSION_4_6),
                           contextResolvingMinMuleVersion);
    verify(mockDeclarer).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarationWhenContextNotResolving() {
    declareWithBaselineMmv(() -> of(resolvedMMV), mockDeclaration, empty(), contextNotResolvingMinMuleVersion);
    verify(mockDeclaration, never()).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarationWhenContextResolving() {
    declareWithBaselineMmv(() -> of(resolvedMMV), mockDeclaration, empty(), contextResolvingMinMuleVersion);
    verify(mockDeclaration).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarationWhenNoResolvedMMV() {
    declareWithBaselineMmv(() -> empty(), mockDeclaration, empty(), contextResolvingMinMuleVersion);
    verify(mockDeclaration, never()).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarationWhenBaselineLower() {
    // Baseline (4.4) is lower than element MMV (4.6), so element MMV should be applied
    declareWithBaselineMmv(() -> of(resolvedMMV), mockDeclaration, of(TEST_VERSION_4_4),
                           contextResolvingMinMuleVersion);
    verify(mockDeclaration).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarationWhenBaselineHigher() {
    // Baseline (4.8) is higher than element MMV (4.6), so element MMV should NOT be applied
    declareWithBaselineMmv(() -> of(resolvedMMV), mockDeclaration, of(TEST_VERSION_4_8),
                           contextResolvingMinMuleVersion);
    verify(mockDeclaration, never()).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarationWhenBaselineEqual() {
    // Baseline (4.6) is equal to element MMV (4.6), baseline is not "newer than", so element MMV should be applied
    declareWithBaselineMmv(() -> of(resolvedMMV), mockDeclaration, of(TEST_VERSION_4_6),
                           contextResolvingMinMuleVersion);
    verify(mockDeclaration).withMinMuleVersion(TEST_VERSION_4_6);
  }

  @Test
  void testDeclareWithBaselineMmvDeclarationWithDefaultVersion() {
    // If resolved MMV is FIRST_MULE_VERSION (4.1.1) and there's no baseline, it should be applied
    ResolvedMinMuleVersion defaultMMV = new ResolvedMinMuleVersion("testComponent", FIRST_MULE_VERSION,
                                                                   "Test component has default min mule version.");
    declareWithBaselineMmv(() -> of(defaultMMV), mockDeclaration, empty(), contextResolvingMinMuleVersion);
    verify(mockDeclaration).withMinMuleVersion(FIRST_MULE_VERSION);
  }
}

