/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.loader.parser.java.source;

import static org.mule.runtime.api.functional.Either.left;
import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.checkInstantiable;

import static java.lang.String.format;

import org.mule.runtime.api.functional.Either;
import org.mule.runtime.extension.api.exception.IllegalSourceModelDefinitionException;
import org.mule.runtime.extension.api.runtime.source.SdkSourceFactory;
import org.mule.runtime.extension.api.runtime.source.SourceFactoryContext;
import org.mule.runtime.module.extension.internal.runtime.source.SourceConfigurer;
import org.mule.runtime.module.extension.internal.util.ReflectionCache;
import org.mule.sdk.api.runtime.source.Source;

public final class JavaSdkSourceFactory implements SdkSourceFactory {

  private final Class<?> sourceType;
  private final boolean isLegacySourceType;

  public JavaSdkSourceFactory(Class<?> sourceType) {
    checkInstantiable(sourceType, new ReflectionCache());
    if (org.mule.runtime.extension.api.runtime.source.Source.class.isAssignableFrom(sourceType)) {
      isLegacySourceType = true;
    } else if (Source.class.isAssignableFrom(sourceType)) {
      isLegacySourceType = false;
    } else {
      throw new IllegalSourceModelDefinitionException(format("Source type %s must extend either %s or %s classes",
                                                             sourceType.getName(),
                                                             Source.class.getName(),
                                                             org.mule.runtime.extension.api.runtime.source.Source.class
                                                                 .getName()));
    }
    this.sourceType = sourceType;
  }

  @Override
  public Either<Source, org.mule.runtime.extension.api.runtime.source.Source> createMessageSource() {
    try {
      Object source = sourceType.getConstructor().newInstance();
      return isLegacySourceType ? right((org.mule.runtime.extension.api.runtime.source.Source<?, ?>) source)
          : left((Source<?, ?>) source);
    } catch (Exception e) {
      throw new RuntimeException("Exception found trying to instantiate source type " + sourceType.getName(), e);
    }
  }

  @Override
  public org.mule.runtime.extension.api.runtime.source.Source<?, ?> createMessageSource(SourceFactoryContext context) {
    try {
      // Creates the source instance
      Object source = sourceType.getConstructor().newInstance();

      // Uses SourceConfigurer to handle parameter injection
      SourceConfigurer sourceConfigurer = new SourceConfigurer(context);

      return sourceConfigurer.configure(source);
    } catch (Exception e) {
      throw new RuntimeException("Exception found trying to instantiate and configure source type " + sourceType.getName(), e);
    }
  }
}
