/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.loader.parser.java;

import static org.mule.runtime.extension.api.values.ValueResolvingException.MISSING_REQUIRED_PARAMETERS;
import static org.mule.runtime.extension.api.values.ValueResolvingException.UNKNOWN;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.setValueIntoField;
import static org.mule.sdk.api.values.ValueResolvingException.CONNECTION_FAILURE;

import static java.lang.String.format;
import static java.util.stream.Collectors.toMap;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.parameter.ActingParameterModel;
import org.mule.runtime.core.api.el.ExpressionManager;
import org.mule.runtime.extension.api.loader.parser.ValueProviderFactory;
import org.mule.runtime.extension.api.loader.parser.ValueProviderFactoryContext;
import org.mule.runtime.extension.api.values.ValueProvider;
import org.mule.runtime.extension.api.values.ValueResolvingException;
import org.mule.runtime.module.extension.api.loader.java.type.FieldElement;
import org.mule.runtime.module.extension.internal.loader.java.property.InjectableParameterInfo;
import org.mule.runtime.module.extension.internal.util.InjectableParameterResolver;
import org.mule.runtime.module.extension.internal.value.SdkValueProviderAdapter;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import jakarta.inject.Inject;

/**
 * {@link ValueProviderFactory} for Java based syntax
 *
 * @since 4.10.0
 */
public class JavaValueProviderFactory implements ValueProviderFactory {

  @Inject
  private ExpressionManager expressionManager;

  private final Class<?> resolverClass;
  private final List<FieldElement> injectableFields;
  private final FieldElement configField;
  private final FieldElement connectionField;

  private final Map<String, MetadataType> fieldTypesByName;

  public JavaValueProviderFactory(Class<?> resolverClass,
                                  List<FieldElement> injectableFields,
                                  FieldElement configField, FieldElement connectionField) {
    this.resolverClass = resolverClass;
    this.injectableFields = injectableFields;
    this.configField = configField;
    this.connectionField = connectionField;
    this.fieldTypesByName = injectableFields.stream()
        .collect(toMap(FieldElement::getName, fe -> fe.getType().asMetadataType()));
  }

  @Override
  public ValueProvider create(ValueProviderFactoryContext context) throws ValueResolvingException {
    Object resolver;
    try {
      resolver = resolverClass.getConstructor().newInstance();
    } catch (ReflectiveOperationException e) {
      throw new ValueResolvingException("An error occurred trying to create a ValueProvider", UNKNOWN, e);
    }

    Map<String, InjectableParameterInfo> injectableParameterInfoByFieldName =
        getInjectableParameterInfos(context.getModel().getParameters());
    InjectableParameterResolver injectableParameterResolver =
        new InjectableParameterResolver(context.getComponentParameterization(),
                                        expressionManager,
                                        injectableParameterInfoByFieldName.values().stream().toList());

    injectValueProviderFields(resolver, injectableParameterResolver, injectableParameterInfoByFieldName);

    if (connectionField != null && connectionField.getField().isPresent()) {

      Object connection;
      if (context.getConnectionSupplier().isEmpty()
          || (connection = getConnection(context.getConnectionSupplier().get())) == null) {
        throw new ValueResolvingException("The value provider requires a connection and none was provided",
                                          MISSING_REQUIRED_PARAMETERS);
      }

      setValueIntoField(resolver, connection, connectionField.getField().get());
    }

    if (configField != null && configField.getField().isPresent()) {
      Object configuration;
      if (context.getConfigurationSupplier().isEmpty()
          || (configuration = context.getConfigurationSupplier().get().get()) == null) {
        throw new ValueResolvingException("The value provider requires a configuration and none was provided",
                                          MISSING_REQUIRED_PARAMETERS);
      }
      setValueIntoField(resolver, configuration, configField.getField().get());
    }

    return adaptResolver(resolver);
  }

  private Object getConnection(Supplier<?> connectionSupplier) throws ValueResolvingException {
    try {
      return connectionSupplier.get();
    } catch (Exception e) {
      throw new ValueResolvingException("Failed to establish connection: " + e.getMessage(), CONNECTION_FAILURE, e);
    }
  }

  public Class<?> getResolverClass() {
    return resolverClass;
  }

  public Map<String, MetadataType> getFieldTypesByName() {
    return fieldTypesByName;
  }

  private void injectValueProviderFields(Object resolver, InjectableParameterResolver injectableParameterResolver,
                                         Map<String, InjectableParameterInfo> injectableParameterInfoByFieldName)
      throws ValueResolvingException {
    List<String> missingParameters = new ArrayList<>();

    for (FieldElement injectableField : injectableFields) {
      InjectableParameterInfo injectableParameterInfo = injectableParameterInfoByFieldName.get(injectableField.getName());
      if (injectableField.getField().isEmpty()) {
        continue;
      }

      String fieldName = injectableParameterInfo.getParameterName();
      Object parameterValue = injectableParameterResolver.getInjectableParameterValue(fieldName);

      if (parameterValue != null) {
        setValueIntoField(resolver, parameterValue, injectableField.getField().get());
      } else if (injectableParameterInfo.isRequired()) {
        String extractionExpression = injectableParameterInfo.getExtractionExpression();
        if (fieldName.equals(extractionExpression)) {
          missingParameters.add(fieldName);
        } else {
          missingParameters.add(fieldName + "(taken from: " + extractionExpression + ")");
        }
      }
    }

    if (!missingParameters.isEmpty()) {
      throw new ValueResolvingException("Unable to retrieve values. There are missing required parameters for the resolution: "
          + missingParameters, MISSING_REQUIRED_PARAMETERS);
    }
  }

  private ValueProvider adaptResolver(Object resolverObject) throws ValueResolvingException {
    if (resolverObject instanceof ValueProvider) {
      return (ValueProvider) resolverObject;
    } else if (resolverObject instanceof org.mule.sdk.api.values.ValueProvider) {
      return new SdkValueProviderAdapter((org.mule.sdk.api.values.ValueProvider) resolverObject);
    } else {
      throw new ValueResolvingException(format("An error occurred trying to create a ValueProvider: %s should implement %s or %s",
                                               resolverObject.getClass().getName(),
                                               ValueProvider.class.getName(),
                                               org.mule.sdk.api.values.ValueProvider.class.getName()),
                                        UNKNOWN);
    }
  }

  Map<String, InjectableParameterInfo> getInjectableParameterInfos(List<ActingParameterModel> actingParameterModels) {
    Map<String, InjectableParameterInfo> injectableParameterInfos = new LinkedHashMap<>();
    int i = 0;
    for (FieldElement fieldElement : injectableFields) {
      injectableParameterInfos.put(fieldElement.getName(),
                                   new InjectableParameterInfo(fieldElement.getName(),
                                                               fieldElement.getType().asMetadataType(),
                                                               fieldElement.isRequired(),
                                                               actingParameterModels.get(i++).getExtractionExpression()));
    }
    return injectableParameterInfos;
  }
}
