/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.config;

import static org.mule.test.allure.AllureConstants.ReuseFeature.REUSE;
import static org.mule.test.allure.AllureConstants.ReuseFeature.ReuseStory.OPERATIONS;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Optional.empty;

import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.Answers.RETURNS_DEEP_STUBS;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mock.Strictness.LENIENT;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.component.Component;
import org.mule.runtime.api.config.ArtifactEncoding;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.metadata.MetadataCache;
import org.mule.runtime.api.metadata.MetadataContext;
import org.mule.runtime.core.api.Injector;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.connector.ConnectionManager;
import org.mule.runtime.core.api.context.notification.ServerNotificationManager;
import org.mule.runtime.core.api.el.ExtendedExpressionManager;
import org.mule.runtime.core.api.streaming.StreamingManager;
import org.mule.runtime.extension.api.runtime.config.ConfigurationInstance;
import org.mule.runtime.module.extension.api.tooling.metadata.MetadataMediator;
import org.mule.runtime.module.extension.api.tooling.sampledata.SampleDataProviderMediator;
import org.mule.runtime.module.extension.api.tooling.valueprovider.ValueProviderMediator;
import org.mule.runtime.module.extension.internal.data.sample.DefaultSampleDataProviderMediator;
import org.mule.runtime.module.extension.internal.metadata.DefaultMetadataContext;
import org.mule.runtime.module.extension.internal.metadata.DefaultMetadataMediator;
import org.mule.runtime.module.extension.internal.util.ReflectionCache;
import org.mule.runtime.module.extension.internal.value.DefaultValueProviderMediator;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

/**
 * Unit tests for {@link DefaultExtensionDesignTimeResolversFactory}.
 *
 * This test class focuses on basic smoke testing of the factory methods to ensure they return non-null instances and perform
 * basic setup. More comprehensive integration testing is done in other test suites.
 */
@SmallTest
@ExtendWith(MockitoExtension.class)
class DefaultExtensionDesignTimeResolversFactoryTestCase extends AbstractMuleTestCase {

  private static final String PARAM_NAME = "paramName";

  private DefaultExtensionDesignTimeResolversFactory factory;

  @Mock(answer = RETURNS_DEEP_STUBS, strictness = LENIENT)
  private MuleContext muleContext;

  @Mock(strictness = LENIENT)
  private ReflectionCache reflectionCache;

  @Mock(strictness = LENIENT)
  private ArtifactEncoding artifactEncoding;

  @Mock(answer = RETURNS_DEEP_STUBS, strictness = LENIENT)
  private ExtendedExpressionManager expressionManager;

  @Mock(answer = RETURNS_DEEP_STUBS, strictness = LENIENT)
  private Registry registry;

  @Mock(answer = RETURNS_DEEP_STUBS, strictness = LENIENT)
  private ParameterizedModel parameterizedModel;

  @Mock(answer = RETURNS_DEEP_STUBS, strictness = LENIENT)
  private ComponentModel componentModel;

  @Mock(strictness = LENIENT)
  private Component component;

  @Mock(strictness = LENIENT)
  private StreamingManager streamingManager;

  @Mock(strictness = LENIENT)
  private ConnectionManager connectionManager;

  @Mock(strictness = LENIENT)
  private MetadataCache metadataCache;

  @Mock(strictness = LENIENT)
  private ClassTypeLoader typeLoader;

  @Mock(strictness = LENIENT)
  private Injector injector;

  @Mock(answer = RETURNS_DEEP_STUBS, strictness = LENIENT)
  private ParameterModel parameterModel;

  @Mock(answer = RETURNS_DEEP_STUBS, strictness = LENIENT)
  private ParameterGroupModel parameterGroupModel;

  @BeforeEach
  void setUp() {
    factory = new DefaultExtensionDesignTimeResolversFactory();

    // Inject dependencies using setters
    factory.setMuleContext(muleContext);
    factory.setReflectionCache(reflectionCache);
    factory.setArtifactEncoding(artifactEncoding);
    factory.setExpressionManager(expressionManager);
    factory.setRegistry(registry);

    // Setup common mocks
    when(artifactEncoding.getDefaultEncoding()).thenReturn(UTF_8);
    when(muleContext.getInjector()).thenReturn(injector);
    when(muleContext.getExecutionClassLoader()).thenReturn(getClass().getClassLoader());

    // Setup parameterized model defaults
    List<ParameterModel> parameterModels = new ArrayList<>();
    parameterModels.add(parameterModel);
    when(parameterizedModel.getAllParameterModels()).thenReturn(parameterModels);
    when(parameterizedModel.getName()).thenReturn(PARAM_NAME);
    when(parameterModel.getName()).thenReturn(PARAM_NAME);
    when(parameterModel.getType()).thenReturn(BaseTypeBuilder.create(MetadataFormat.JAVA).stringType().build());

    List<ParameterGroupModel> parameterGroupModels = new ArrayList<>();
    parameterGroupModels.add(parameterGroupModel);
    when(parameterizedModel.getParameterGroupModels()).thenReturn(parameterGroupModels);
    when(parameterGroupModel.getParameterModels()).thenReturn(parameterModels);
    when(parameterGroupModel.getName()).thenReturn("General");

    when(muleContext.getNotificationManager())
        .thenReturn(mock(ServerNotificationManager.class));
  }

  @Test
  void createValueProviderMediator() {
    ValueProviderMediator mediator = factory.createValueProviderMediator(parameterizedModel);

    assertThat(mediator, is(notNullValue()));
    assertThat(mediator, is(instanceOf(DefaultValueProviderMediator.class)));
  }

  @Test
  void createSampleDataProviderMediator() {
    SampleDataProviderMediator mediator = factory.createSampleDataProviderMediator(
                                                                                   mock(ExtensionModel.class),
                                                                                   componentModel,
                                                                                   component,
                                                                                   streamingManager);

    assertThat(mediator, is(notNullValue()));
    assertThat(mediator, is(instanceOf(DefaultSampleDataProviderMediator.class)));
  }

  @Test
  void createMetadataContextBasic() {
    Supplier<Optional<ConfigurationInstance>> configSupplier = () -> empty();

    MetadataContext context = factory.createMetadataContext(
                                                            configSupplier,
                                                            connectionManager,
                                                            metadataCache,
                                                            typeLoader);

    assertThat(context, is(notNullValue()));
    assertThat(context, is(instanceOf(DefaultMetadataContext.class)));
  }

  @Test
  void createMetadataContextWithOutputContexts() {
    Supplier<Optional<ConfigurationInstance>> configSupplier = () -> empty();

    MetadataContext context = factory.createMetadataContext(
                                                            configSupplier,
                                                            connectionManager,
                                                            metadataCache,
                                                            typeLoader,
                                                            empty(),
                                                            empty());

    assertThat(context, is(notNullValue()));
    assertThat(context, is(instanceOf(DefaultMetadataContext.class)));
  }

  @Test
  void createMetadataMediator() {
    MetadataMediator mediator = factory.createMetadataMediator(componentModel);

    assertThat(mediator, is(notNullValue()));
    assertThat(mediator, is(instanceOf(DefaultMetadataMediator.class)));
  }

  @Test
  void createMetadataContextWithConfigurationInstance() {
    ConfigurationInstance configInstance = mock(ConfigurationInstance.class);
    Supplier<Optional<ConfigurationInstance>> configSupplier = () -> Optional.of(configInstance);

    MetadataContext context = factory.createMetadataContext(
                                                            configSupplier,
                                                            connectionManager,
                                                            metadataCache,
                                                            typeLoader);

    assertThat(context, is(notNullValue()));
  }

  @Test
  void verifyRegistryLookupIsUsed() throws Exception {
    when(registry.lookupByType(any())).thenReturn(empty());

    // Call a method that uses the registry
    factory.createValueProviderMediator(parameterizedModel);

    // Verify the factory has the registry injected and can use it
    assertThat(factory, is(notNullValue()));
  }
}
