/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.transaction;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.Mockito.when;

import org.mule.runtime.extension.api.runtime.config.ConfigurationInstance;
import org.mule.runtime.module.extension.internal.runtime.operation.ExecutionContextConfigurationDecorator;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

@SmallTest
@ExtendWith(MockitoExtension.class)
class ExtensionTransactionKeyTestCase extends AbstractMuleTestCase {

  @Mock
  private ConfigurationInstance configurationInstance;

  @Mock
  private ConfigurationInstance anotherConfigurationInstance;

  @Mock
  private ConfigurationInstance decoratedConfigurationInstance;

  @Mock
  private ExecutionContextConfigurationDecorator decoratorConfigurationInstance;

  @Mock
  private ExecutionContextConfigurationDecorator outerDecoratorConfigurationInstance;

  private ExtensionTransactionKey key;

  @BeforeEach
  public void setUp() {
    key = new ExtensionTransactionKey(configurationInstance);
  }

  @Test
  void constructorWithRegularConfiguration() {
    ExtensionTransactionKey transactionKey = new ExtensionTransactionKey(configurationInstance);

    assertThat(transactionKey, is(notNullValue()));
  }

  @Test
  void constructorWithDecoratedConfiguration() {
    when(decoratorConfigurationInstance.getDecorated()).thenReturn(decoratedConfigurationInstance);

    ExtensionTransactionKey transactionKey = new ExtensionTransactionKey(decoratorConfigurationInstance);

    assertThat(transactionKey, is(notNullValue()));
  }

  @Test
  void equalsReflexive() {
    // An object should be equal to itself
    assertThat(key.equals(key), is(true));
  }

  @Test
  void equalsSymmetric() {
    // If x.equals(y), then y.equals(x)
    ExtensionTransactionKey key1 = new ExtensionTransactionKey(configurationInstance);
    ExtensionTransactionKey key2 = new ExtensionTransactionKey(configurationInstance);

    assertThat(key1.equals(key2), is(key2.equals(key1)));
  }

  @Test
  void equalsTransitive() {
    // If x.equals(y) and y.equals(z), then x.equals(z)
    ExtensionTransactionKey key1 = new ExtensionTransactionKey(configurationInstance);
    ExtensionTransactionKey key2 = new ExtensionTransactionKey(configurationInstance);
    ExtensionTransactionKey key3 = new ExtensionTransactionKey(configurationInstance);

    assertThat(key1.equals(key2), is(true));
    assertThat(key2.equals(key3), is(true));
    assertThat(key1.equals(key3), is(true));
  }

  @Test
  void equalsConsistent() {
    // Multiple invocations should return the same result
    ExtensionTransactionKey key1 = new ExtensionTransactionKey(configurationInstance);
    ExtensionTransactionKey key2 = new ExtensionTransactionKey(configurationInstance);

    assertThat(key1.equals(key2), is(true));
    assertThat(key1.equals(key2), is(true));
    assertThat(key1.equals(key2), is(true));
  }

  @Test
  void equalsWithNull() {
    assertThat(key.equals(null), is(false));
  }

  @Test
  void equalsWithDifferentType() {
    assertThat(key.equals("not a transaction key"), is(false));
  }

  @Test
  void equalsWithSameConfiguration() {
    ExtensionTransactionKey key1 = new ExtensionTransactionKey(configurationInstance);
    ExtensionTransactionKey key2 = new ExtensionTransactionKey(configurationInstance);

    assertThat(key1, is(equalTo(key2)));
  }

  @Test
  void equalsWithDifferentConfiguration() {
    ExtensionTransactionKey key1 = new ExtensionTransactionKey(configurationInstance);
    ExtensionTransactionKey key2 = new ExtensionTransactionKey(anotherConfigurationInstance);

    assertThat(key1, is(not(equalTo(key2))));
  }

  @Test
  void equalsWithDecoratedConfiguration() {
    when(decoratorConfigurationInstance.getDecorated()).thenReturn(decoratedConfigurationInstance);

    ExtensionTransactionKey key1 = new ExtensionTransactionKey(decoratorConfigurationInstance);
    ExtensionTransactionKey key2 = new ExtensionTransactionKey(decoratedConfigurationInstance);

    assertThat(key1, is(equalTo(key2)));
  }

  @Test
  void equalsWithDecoratorAndRegularConfigurationSameInstance() {
    when(decoratorConfigurationInstance.getDecorated()).thenReturn(configurationInstance);

    ExtensionTransactionKey key1 = new ExtensionTransactionKey(decoratorConfigurationInstance);
    ExtensionTransactionKey key2 = new ExtensionTransactionKey(configurationInstance);

    assertThat(key1, is(equalTo(key2)));
  }

  @Test
  void hashCodeConsistency() {
    // Multiple invocations should return the same value
    int hashCode1 = key.hashCode();
    int hashCode2 = key.hashCode();
    int hashCode3 = key.hashCode();

    assertThat(hashCode1, is(hashCode2));
    assertThat(hashCode2, is(hashCode3));
  }

  @Test
  void hashCodeEqualObjectsHaveEqualHashCodes() {
    // If two objects are equal, they must have the same hash code
    ExtensionTransactionKey key1 = new ExtensionTransactionKey(configurationInstance);
    ExtensionTransactionKey key2 = new ExtensionTransactionKey(configurationInstance);

    assertThat(key1, is(equalTo(key2)));
    assertThat(key1.hashCode(), is(key2.hashCode()));
  }

  @Test
  void hashCodeWithSameConfiguration() {
    ExtensionTransactionKey key1 = new ExtensionTransactionKey(configurationInstance);
    ExtensionTransactionKey key2 = new ExtensionTransactionKey(configurationInstance);

    assertThat(key1.hashCode(), is(key2.hashCode()));
  }

  @Test
  void hashCodeWithDifferentConfiguration() {
    ExtensionTransactionKey key1 = new ExtensionTransactionKey(configurationInstance);
    ExtensionTransactionKey key2 = new ExtensionTransactionKey(anotherConfigurationInstance);

    // Different objects may or may not have different hash codes, but we expect them to be different
    assertThat(key1.hashCode(), is(not(key2.hashCode())));
  }

  @Test
  void hashCodeWithDecoratedConfiguration() {
    when(decoratorConfigurationInstance.getDecorated()).thenReturn(decoratedConfigurationInstance);

    ExtensionTransactionKey key1 = new ExtensionTransactionKey(decoratorConfigurationInstance);
    ExtensionTransactionKey key2 = new ExtensionTransactionKey(decoratedConfigurationInstance);

    assertThat(key1.hashCode(), is(key2.hashCode()));
  }

  @Test
  void hashCodeWithDecoratorAndRegularConfigurationSameInstance() {
    when(decoratorConfigurationInstance.getDecorated()).thenReturn(configurationInstance);

    ExtensionTransactionKey key1 = new ExtensionTransactionKey(decoratorConfigurationInstance);
    ExtensionTransactionKey key2 = new ExtensionTransactionKey(configurationInstance);

    assertThat(key1.hashCode(), is(key2.hashCode()));
  }

  @Test
  void decoratorUnwrapsToInnerConfiguration() {
    when(decoratorConfigurationInstance.getDecorated()).thenReturn(decoratedConfigurationInstance);

    ExtensionTransactionKey keyWithDecorator = new ExtensionTransactionKey(decoratorConfigurationInstance);
    ExtensionTransactionKey keyWithInner = new ExtensionTransactionKey(decoratedConfigurationInstance);

    assertThat(keyWithDecorator, is(equalTo(keyWithInner)));
    assertThat(keyWithDecorator.hashCode(), is(keyWithInner.hashCode()));
  }

  @Test
  void decoratorUnwrapsOneLevelOnly() {
    // Setup: decorator wraps regular config
    when(decoratorConfigurationInstance.getDecorated()).thenReturn(configurationInstance);

    // When creating a key from the decorator, it unwraps one level to configurationInstance
    ExtensionTransactionKey keyFromDecorator = new ExtensionTransactionKey(decoratorConfigurationInstance);
    ExtensionTransactionKey keyFromRegularConfig = new ExtensionTransactionKey(configurationInstance);

    // The decorator unwraps to the regular config, so these should be equal
    assertThat(keyFromDecorator, is(equalTo(keyFromRegularConfig)));
  }
}

