/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.resolver;

import static org.mule.metadata.api.model.MetadataFormat.JSON;
import static org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED;
import static org.mule.runtime.api.metadata.DataType.JSON_STRING;
import static org.mule.runtime.core.api.event.CoreEvent.nullEvent;
import static org.mule.runtime.module.extension.internal.runtime.resolver.ResolverSetUtils.getResolverSetFromParameters;

import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsMapContaining.hasEntry;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsInstanceOf.instanceOf;
import static org.hamcrest.core.IsNull.nullValue;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mock.Strictness.LENIENT;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.meta.model.ParameterDslConfiguration;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.parameterization.ComponentParameterization;
import org.mule.runtime.api.transformation.TransformationService;
import org.mule.runtime.core.api.Injector;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.el.ExtendedExpressionManager;
import org.mule.runtime.core.internal.config.DefaultArtifactEncoding;
import org.mule.runtime.extension.api.declaration.type.annotation.ParameterResolverTypeAnnotation;
import org.mule.runtime.extension.api.declaration.type.annotation.TypedValueTypeAnnotation;
import org.mule.runtime.module.extension.api.runtime.resolver.ResolverSet;
import org.mule.runtime.module.extension.api.runtime.resolver.ValueResolver;
import org.mule.runtime.module.extension.api.runtime.resolver.ValueResolvingContext;
import org.mule.runtime.module.extension.internal.runtime.resolver.resolver.ValueResolverFactory;
import org.mule.runtime.module.extension.internal.util.ReflectionCache;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit5.InjectDataWeave;
import org.mule.test.module.extension.internal.util.ExtensionsTestUtils;

import java.util.Map;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

import io.qameta.allure.Issue;

@ExtendWith(MockitoExtension.class)
public class ResolverSetUtilsTestCase extends AbstractMuleTestCase {

  @InjectDataWeave
  private ExtendedExpressionManager expressionManager;

  @Mock
  private MuleContext muleContext;

  @Mock(strictness = LENIENT)
  private Injector injector;

  @Mock
  private TransformationService transformationService;

  @BeforeEach
  public void setUp() throws MuleException {
    when(muleContext.getInjector()).thenReturn(injector);
    when(injector.inject(any(TypeSafeValueResolverWrapper.class))).thenAnswer(inv -> {
      inv.getArgument(0, TypeSafeValueResolverWrapper.class).setTransformationService(transformationService);
      return null;
    });
    when(injector.inject(any(TypeSafeExpressionValueResolver.class))).thenAnswer(inv -> {
      inv.getArgument(0, TypeSafeExpressionValueResolver.class).setTransformationService(transformationService);
      return null;
    });
    when(injector.inject(any(ExpressionBasedParameterResolverValueResolver.class))).thenAnswer(inv -> {
      inv.getArgument(0, ExpressionBasedParameterResolverValueResolver.class).setTransformationService(transformationService);
      return null;
    });
    when(injector.inject(any(ExpressionTypedValueValueResolver.class))).thenAnswer(inv -> {
      inv.getArgument(0, ExpressionTypedValueValueResolver.class).setTransformationService(transformationService);
      return null;
    });
  }

  @Test
  @Issue("W-17579307")
  void pojoFromComponentParameterization() throws MuleException {
    MetadataType pojoType = ExtensionsTestUtils.TYPE_LOADER.load(MyPojo.class);
    ParameterizedModel pmzdModel = createModelWithParam(pojoType);
    ParameterizedModel pojoModel = createPojoModel();

    ComponentParameterization<ParameterizedModel> componentParameterization =
        ComponentParameterization.builder(pojoModel)
            .withParameter("textValue", "hello me")
            .withParameter("numberValue", 42)
            .build();

    ResolverSet resolvers = getResolverSetFromParameters(pmzdModel,
                                                         (pgm, pm) -> componentParameterization,
                                                         muleContext,
                                                         true,
                                                         new ReflectionCache(),
                                                         expressionManager, "",
                                                         new ValueResolverFactory(), new DefaultArtifactEncoding(null));
    ValueResolver<?> paramResolver = resolvers.getResolvers().get("paramName");
    Object resolvedValue = paramResolver.resolve(null);

    assertThat(resolvedValue, instanceOf(MyPojo.class));
    assertThat(((MyPojo) resolvedValue).getTextValue(), is("hello me"));
    assertThat(((MyPojo) resolvedValue).getNumberValue(), is(42));
  }

  @Test
  @Issue("W-18030082")
  void nonJavaObject() throws MuleException {
    MetadataType jsonType = BaseTypeBuilder.create(JSON)
        .stringType()
        .build();

    ParameterizedModel pmzdModel = createModelWithParam(jsonType);
    Object value = "{key: 'value'}";

    ResolverSet resolvers = getResolverSetFromParameters(pmzdModel,
                                                         (pgm, pm) -> value,
                                                         muleContext,
                                                         true,
                                                         new ReflectionCache(),
                                                         expressionManager, "",
                                                         new ValueResolverFactory(), new DefaultArtifactEncoding(null));

    ValueResolver<?> paramResolver = resolvers.getResolvers().get("paramName");
    Object resolvedValue = paramResolver.resolve(null);

    assertThat(resolvedValue, instanceOf(String.class));
    assertThat(resolvedValue, is("{key: 'value'}"));
  }

  @Test
  @Issue("W-18030082")
  void nonJavaTypedValue() throws MuleException {
    MetadataType jsonType = BaseTypeBuilder.create(JSON)
        .stringType()
        .build();

    ParameterizedModel pmzdModel = createModelWithParam(jsonType);
    TypedValue value = new TypedValue<>("{key: 'value'}", JSON_STRING);

    ResolverSet resolvers = getResolverSetFromParameters(pmzdModel,
                                                         (pgm, pm) -> value,
                                                         muleContext,
                                                         true,
                                                         new ReflectionCache(),
                                                         expressionManager, "",
                                                         new ValueResolverFactory(), new DefaultArtifactEncoding(null));

    ValueResolver<?> paramResolver = resolvers.getResolvers().get("paramName");
    Object resolvedValue = paramResolver.resolve(null);

    assertThat(resolvedValue, instanceOf(String.class));
    assertThat(resolvedValue, is("{key: 'value'}"));
  }

  @Test
  @Issue("W-18030082")
  void nonJavaExpressionTypedValue() throws MuleException {
    when(transformationService.transform(any(), any(), any())).thenReturn("hello me");

    MetadataType jsonType = BaseTypeBuilder.create(JSON)
        .stringType()
        .build();

    ParameterizedModel pmzdModel = createModelWithExpressionParam(jsonType);

    String value = "#['hello me']";

    ResolverSet resolvers = getResolverSetFromParameters(pmzdModel,
                                                         (pgm, pm) -> value,
                                                         muleContext,
                                                         true,
                                                         new ReflectionCache(),
                                                         expressionManager, "",
                                                         new ValueResolverFactory(), new DefaultArtifactEncoding(null));

    ValueResolver<?> paramResolver = resolvers.getResolvers().get("paramName");
    Object resolvedValue = paramResolver.resolve(null);

    assertThat(resolvedValue, instanceOf(String.class));
    assertThat(resolvedValue, is("hello me"));
  }

  @Test
  @Issue("W-18030082")
  void nonJavaDefaultExpressionTypedValue() throws MuleException {
    MetadataType jsonType = BaseTypeBuilder.create(JSON)
        .stringType()
        .build();

    ParameterizedModel pmzdModel = createModelWithExpressionParam(jsonType);

    String value = "#[payload]";

    ResolverSet resolvers = getResolverSetFromParameters(pmzdModel,
                                                         (pgm, pm) -> value,
                                                         muleContext,
                                                         true,
                                                         new ReflectionCache(),
                                                         expressionManager, "",
                                                         new ValueResolverFactory(), new DefaultArtifactEncoding(null));

    ValueResolver<?> paramResolver = resolvers.getResolvers().get("paramName");
    Object resolvedValue = paramResolver.resolve(null);

    assertThat(resolvedValue, is(nullValue()));
  }

  @Test
  @Issue("W-20187776")
  void objectTypeWithoutClassAndExpressionDefault() throws MuleException {
    MetadataType jsonType = BaseTypeBuilder.create(JSON)
        .objectType()
        .build();

    ParameterizedModel pmzdModel = createModelWithExpressionParamAndDefault(jsonType, "#['someKey': 'someValue']");

    String value = null;

    ResolverSet resolvers = getResolverSetFromParameters(pmzdModel,
                                                         (pgm, pm) -> value,
                                                         muleContext,
                                                         true,
                                                         new ReflectionCache(),
                                                         expressionManager, "",
                                                         new ValueResolverFactory(), new DefaultArtifactEncoding(null));

    ValueResolver<?> paramResolver = resolvers.getResolvers().get("paramName");
    Object resolvedValue = paramResolver.resolve(ValueResolvingContext.builder(nullEvent()).build());

    assertThat((Map<String, String>) resolvedValue, hasEntry("someKey", "someValue"));
  }

  @Test
  @Issue("W-20187776")
  void typedValueObjectTypeWithoutClassAndExpressionDefault() throws MuleException {
    MetadataType jsonType = BaseTypeBuilder.create(JSON)
        .objectType()
        .with(new TypedValueTypeAnnotation())
        .build();

    ParameterizedModel pmzdModel = createModelWithExpressionParamAndDefault(jsonType, "#['someKey': 'someValue']");

    String value = null;

    ResolverSet resolvers = getResolverSetFromParameters(pmzdModel,
                                                         (pgm, pm) -> value,
                                                         muleContext,
                                                         true,
                                                         new ReflectionCache(),
                                                         expressionManager, "",
                                                         new ValueResolverFactory(), new DefaultArtifactEncoding(null));

    ValueResolver<?> paramResolver = resolvers.getResolvers().get("paramName");
    TypedValue<Map<String, String>> resolvedValue =
        (TypedValue<Map<String, String>>) paramResolver.resolve(ValueResolvingContext.builder(nullEvent()).build());

    assertThat(resolvedValue.getValue(), hasEntry("someKey", "someValue"));
  }

  @Test
  @Issue("W-20187776")
  void parameterResolverObjectTypeWithoutClassAndExpressionDefault() throws MuleException {
    MetadataType jsonType = BaseTypeBuilder.create(JSON)
        .objectType()
        .with(new ParameterResolverTypeAnnotation())
        .build();

    ParameterizedModel pmzdModel = createModelWithExpressionParamAndDefault(jsonType, "#['someKey': 'someValue']");

    String value = null;

    ResolverSet resolvers = getResolverSetFromParameters(pmzdModel,
                                                         (pgm, pm) -> value,
                                                         muleContext,
                                                         true,
                                                         new ReflectionCache(),
                                                         expressionManager, "",
                                                         new ValueResolverFactory(), new DefaultArtifactEncoding(null));

    ValueResolver<?> paramResolver = resolvers.getResolvers().get("paramName");
    ExpressionBasedParameterResolver<Map<String, String>> resolvedValue =
        (ExpressionBasedParameterResolver<Map<String, String>>) paramResolver
            .resolve(ValueResolvingContext.builder(nullEvent()).build());

    assertThat(resolvedValue.getExpression().get(), is("#['someKey': 'someValue']"));
    assertThat(resolvedValue.resolve(), hasEntry("someKey", "someValue"));
  }

  private ParameterizedModel createModelWithParam(MetadataType type) {
    ParameterModel paramModel = mock(ParameterModel.class);
    when(paramModel.getName()).thenReturn("paramName");
    when(paramModel.getType()).thenReturn(type);
    when(paramModel.getModelProperties()).thenReturn(emptySet());
    when(paramModel.getDslConfiguration()).thenReturn(new ParameterDslConfiguration());

    return createModelWithParameter(paramModel);
  }

  private ParameterizedModel createModelWithExpressionParam(MetadataType type) {
    ParameterModel paramModel = mock(ParameterModel.class);
    when(paramModel.getName()).thenReturn("paramName");
    when(paramModel.getType()).thenReturn(type);
    when(paramModel.getExpressionSupport()).thenReturn(SUPPORTED);
    when(paramModel.getModelProperties()).thenReturn(emptySet());
    when(paramModel.getDslConfiguration()).thenReturn(new ParameterDslConfiguration());

    return createModelWithParameter(paramModel);
  }

  private ParameterizedModel createModelWithExpressionParamAndDefault(MetadataType type, Object defaultValue) {
    ParameterModel paramModel = mock(ParameterModel.class);
    when(paramModel.getName()).thenReturn("paramName");
    when(paramModel.getType()).thenReturn(type);
    when(paramModel.getModelProperties()).thenReturn(emptySet());
    when(paramModel.getDefaultValue()).thenReturn(defaultValue);

    return createModelWithParameter(paramModel);
  }

  private ParameterizedModel createModelWithParameter(ParameterModel parameterModel) {
    ParameterGroupModel paramGroupModel = mock(ParameterGroupModel.class);
    when(paramGroupModel.getParameterModels()).thenReturn(singletonList(parameterModel));

    ParameterizedModel pmzdModel = mock(ParameterizedModel.class);
    when(pmzdModel.getParameterGroupModels()).thenReturn(singletonList(paramGroupModel));
    when(pmzdModel.getAllParameterModels()).thenReturn(singletonList(parameterModel));
    return pmzdModel;
  }

  private ParameterizedModel createPojoModel() {
    ParameterModel pojoTextParamModel = mock(ParameterModel.class);
    when(pojoTextParamModel.getName()).thenReturn("textValue");

    ParameterModel pojoNumberParamModel = mock(ParameterModel.class);
    when(pojoNumberParamModel.getName()).thenReturn("numberValue");

    ParameterGroupModel pojoParamGroupModel = mock(ParameterGroupModel.class);
    when(pojoParamGroupModel.getName()).thenReturn("MyPojo");
    when(pojoParamGroupModel.getParameter("textValue")).thenReturn(of(pojoTextParamModel));
    when(pojoParamGroupModel.getParameter("numberValue")).thenReturn(of(pojoNumberParamModel));

    ParameterizedModel pojoModel = mock(ParameterizedModel.class);
    when(pojoModel.getParameterGroupModels()).thenReturn(singletonList(pojoParamGroupModel));
    return pojoModel;
  }

  public static class MyPojo {

    private String textValue;

    private int numberValue;

    public String getTextValue() {
      return textValue;
    }

    public void setTextValue(String textValue) {
      this.textValue = textValue;
    }

    public int getNumberValue() {
      return numberValue;
    }

    public void setNumberValue(int numberValue) {
      this.numberValue = numberValue;
    }

  }
}
