/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.source;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.module.extension.internal.runtime.resolver.ResolverUtils.resolveCursorAsUnclosable;
import static org.mule.runtime.module.extension.internal.runtime.source.legacy.LegacySourceAdapterFactory.createAdapter;
import static org.mule.runtime.module.extension.internal.runtime.source.legacy.SourceTransactionalActionUtils.toLegacy;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.checkInstantiable;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.fetchConfigFieldFromSourceObject;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.fetchConnectionFieldFromSourceObject;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.getField;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.getImplementingName;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.injectComponentLocation;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.injectDefaultEncoding;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.injectRefName;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.injectRuntimeVersion;

import static java.lang.String.format;

import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.model.EnrichableModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.declaration.type.TypeUtils;
import org.mule.runtime.extension.api.exception.IllegalModelDefinitionException;
import org.mule.runtime.extension.api.runtime.config.ConfigurationInstance;
import org.mule.runtime.extension.api.runtime.source.Source;
import org.mule.runtime.extension.api.runtime.source.SourceFactoryContext;
import org.mule.runtime.extension.api.tx.SourceTransactionalAction;
import org.mule.runtime.module.extension.api.loader.java.type.FieldElement;
import org.mule.runtime.module.extension.internal.loader.ParameterGroupDescriptor;
import org.mule.runtime.module.extension.internal.loader.java.property.ParameterGroupModelProperty;
import org.mule.runtime.module.extension.internal.loader.parser.java.connection.ReverseSdkConnectionProviderAdapter;
import org.mule.runtime.module.extension.internal.util.FieldSetter;
import org.mule.runtime.module.extension.internal.util.ReflectionCache;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Optional;

/**
 * Resolves and injects the values of a {@link Source} that has fields annotated with {@link Parameter} or
 * {@link ParameterGroupDescriptor}
 *
 * @since 4.0
 */
public final class SourceConfigurer {

  private final SourceFactoryContext context;
  private final ReflectionCache reflectionCache = new ReflectionCache();

  /**
   * Create a new instance
   *
   * @param context the {@link SourceFactoryContext}
   */
  public SourceConfigurer(SourceFactoryContext context) {
    this.context = context;
  }

  /**
   * Performs the configuration of the given {@code source} and returns the result
   *
   * @param source a {@link Source}
   * @return the configured instance
   */
  public Source<?, ?> configure(Object source) {
    ParameterizedModel model = context.getParameterization().getModel();
    for (ParameterGroupModel pgm : model.getParameterGroupModels()) {
      if (pgm.getName().equals(DEFAULT_GROUP_NAME)) {
        for (ParameterModel pm : pgm.getParameterModels()) {
          getField(source.getClass(), getImplementingName(pm), reflectionCache).ifPresent(field -> {
            Object fieldValue = context.getParameterization().getParameter(pgm, pm);
            if (fieldValue != null) {
              new FieldSetter<>(field).set(source, adaptToLegacyIfNeeded(fieldValue, field.getType()));
            }
          });
        }
      } else {
        // Non default group
        pgm.getModelProperty(ParameterGroupModelProperty.class)
            .map(ParameterGroupModelProperty::getDescriptor)
            .ifPresent(desc -> {
              if (desc.getContainer() instanceof Field field) {
                Class<?> prototypeClass = desc.getType().getDeclaringClass().get();
                checkInstantiable(prototypeClass, reflectionCache);
                List<FieldElement> groupDescriptorFields = reflectionCache.fieldElementsFor(desc).stream()
                    .filter(fieldElement -> fieldElement.getField().map(TypeUtils::isParameter).orElse(false))
                    .toList();

                Object fieldValue = createInstance(prototypeClass);

                for (FieldElement groupField : groupDescriptorFields) {
                  Object resolvedValue = context.getParameterization().getParameter(pgm.getName(), groupField.getAlias());
                  if (resolvedValue != null) {
                    Object value = resolveCursorAsUnclosable(resolvedValue);
                    groupField.set(fieldValue, adaptToLegacyIfNeeded(value, field.getType()));
                  }
                }

                new FieldSetter<>(field).set(source, fieldValue);
              }
            });
      }
    }

    if (model instanceof EnrichableModel enrichableModel) {
      injectDefaultEncoding(enrichableModel, source, context.getDefaultEncoding());
      injectRuntimeVersion(enrichableModel, source, context.getRuntimeVersion());
    }
    injectComponentLocation(source, context.getComponentLocation());
    context.getConfigurationInstance().ifPresent(configurationInstance -> {
      injectRefName(source, configurationInstance.getName(), reflectionCache);
      setConfiguration(source, configurationInstance);
    });

    context.getConnectionProvider()
        .ifPresent(connectionProvider -> setConnection(source, connectionProvider));

    // adapt if legacy
    return createAdapter(source);
  }

  private Object adaptToLegacyIfNeeded(Object value, Class<?> targetType) {
    if (targetType.isAssignableFrom(SourceTransactionalAction.class)) {
      return toLegacy(value);
    }
    return value;
  }

  private static <T> T createInstance(Class<T> prototypeClass) {
    try {
      return prototypeClass.getConstructor().newInstance();
    } catch (InvocationTargetException e) {
      throw new MuleRuntimeException(createStaticMessage("Could not create instance of " + prototypeClass),
                                     e.getTargetException());
    } catch (Exception e) {
      throw new MuleRuntimeException(createStaticMessage("Could not create instance of " + prototypeClass), e);
    }
  }

  private Optional<Field> fetchConnectionProviderField(Object source) {
    Optional<Field> connectionField = fetchConnectionFieldFromSourceObject(source);

    // Validates field type
    connectionField
        .filter(field -> !fieldAcceptsConnectionProvider(field))
        .ifPresent(field -> {
          throw new IllegalModelDefinitionException(format(
                                                           "Message Source defined on class '%s' has field '%s' of type '%s' annotated with @%s. That annotation can only be "
                                                               + "used on fields of type '%s' or '%s'",
                                                           source.getClass().getName(), field.getName(),
                                                           field.getType().getName(),
                                                           Connection.class.getSimpleName(),
                                                           ConnectionProvider.class.getName(),
                                                           org.mule.sdk.api.connectivity.ConnectionProvider.class.getName()));
        });

    return connectionField;
  }

  private boolean fieldAcceptsConnectionProvider(Field field) {
    return ConnectionProvider.class.equals(field.getType()) ||
        org.mule.sdk.api.connectivity.ConnectionProvider.class.equals(field.getType());
  }

  private void setConfiguration(Object source, ConfigurationInstance configuration) {
    Optional<Field> configurationField = fetchConfigFieldFromSourceObject(source);
    configurationField.ifPresent(field -> new FieldSetter<>(field).set(source, configuration.getValue()));
  }

  private void setConnection(Object source, ConnectionProvider<?> connectionProvider) {
    fetchConnectionProviderField(source).ifPresent(field -> {
      Object adaptedConnectionProvider = connectionProvider;
      if (org.mule.sdk.api.connectivity.ConnectionProvider.class.isAssignableFrom(field.getType())) {
        adaptedConnectionProvider = ReverseSdkConnectionProviderAdapter.from(connectionProvider);
      }
      new FieldSetter<>(field).set(source, adaptedConnectionProvider);
    });
  }
}
