/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.config;

import static org.mule.test.module.extension.internal.util.ExtensionsTestUtils.mockParameters;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.concurrent.TimeUnit.MINUTES;

import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.MatcherAssert.assertThat;

import static org.mockito.Answers.RETURNS_DEEP_STUBS;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mock.Strictness.LENIENT;
import static org.mockito.Mockito.atLeast;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.lenient;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.api.config.ArtifactEncoding;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.parameterization.ComponentParameterization;
import org.mule.runtime.core.api.Injector;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.el.ExtendedExpressionManager;
import org.mule.runtime.core.internal.config.ImmutableExpirationPolicy;
import org.mule.runtime.extension.api.connectivity.oauth.AuthorizationCodeGrantType;
import org.mule.runtime.extension.api.connectivity.oauth.ClientCredentialsGrantType;
import org.mule.runtime.extension.api.connectivity.oauth.OAuthGrantTypeVisitor;
import org.mule.runtime.extension.api.connectivity.oauth.OAuthModelProperty;
import org.mule.runtime.extension.api.connectivity.oauth.PlatformManagedOAuthGrantType;
import org.mule.runtime.extension.api.dsl.syntax.resolver.DslSyntaxResolver;
import org.mule.runtime.extension.api.runtime.config.ConfigurationProvider;
import org.mule.runtime.module.extension.api.runtime.config.ConfigurationProviderFactory;
import org.mule.runtime.module.extension.api.runtime.resolver.ConnectionProviderValueResolver;
import org.mule.runtime.module.extension.api.runtime.resolver.ResolverSet;
import org.mule.runtime.extension.api.runtime.connectivity.ConnectionProviderFactory;
import org.mule.runtime.module.extension.internal.loader.java.property.ConnectionProviderFactoryModelProperty;
import org.mule.runtime.module.extension.internal.runtime.connectivity.ConnectionProviderSettings;
import org.mule.runtime.module.extension.internal.runtime.connectivity.oauth.authcode.AuthorizationCodeOAuthHandler;
import org.mule.runtime.module.extension.internal.runtime.connectivity.oauth.clientcredentials.ClientCredentialsOAuthHandler;
import org.mule.runtime.module.extension.internal.runtime.connectivity.oauth.ocs.PlatformManagedOAuthHandler;
import org.mule.runtime.module.extension.internal.runtime.resolver.ConnectionProviderResolver;
import org.mule.runtime.module.extension.internal.util.ReflectionCache;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;
import org.mule.tck.util.TestTimeSupplier;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

@SmallTest
@ExtendWith(MockitoExtension.class)
class ConfigurationCreationUtilsTestCase extends AbstractMuleTestCase {

  private static final String CONFIG_NAME = "testConfig";
  private static final String EXTENSION_NAME = "testExtension";
  private static final String PARAMETERS_OWNER = "testOwner";

  @Mock(strictness = LENIENT)
  private ExtensionModel extensionModel;

  @Mock(answer = RETURNS_DEEP_STUBS, strictness = LENIENT)
  private ConfigurationModel configurationModel;

  @Mock(answer = RETURNS_DEEP_STUBS, strictness = LENIENT)
  private ConnectionProviderModel connectionProviderModel;

  @Mock(answer = RETURNS_DEEP_STUBS, strictness = LENIENT)
  private MuleContext muleContext;

  @Mock(strictness = LENIENT)
  private ExtendedExpressionManager expressionManager;

  @Mock(strictness = LENIENT)
  private ConfigurationProviderFactory configurationProviderFactory;

  @Mock(strictness = LENIENT)
  private DslSyntaxResolver dslSyntaxResolver;

  @Mock(strictness = LENIENT)
  private ConfigurationProperties configurationProperties;

  @Mock(strictness = LENIENT)
  private ConnectionProviderValueResolver<Object> connectionProviderValueResolver;

  @Mock(strictness = LENIENT)
  private ComponentParameterization<ConnectionProviderModel> componentParameterization;

  @Mock(strictness = LENIENT)
  private ConnectionProviderSettings connectionProviderSettings;

  @Mock(strictness = LENIENT)
  private ConfigurationProvider staticConfigurationProvider;

  @Mock(strictness = LENIENT)
  private ConfigurationProvider dynamicConfigurationProvider;

  @Mock(strictness = LENIENT)
  private Injector injector;

  @Mock(strictness = LENIENT)
  private ErrorTypeRepository errorTypeRepository;

  private ReflectionCache reflectionCache;
  private ArtifactEncoding artifactEncoding;
  private ClassLoader extensionClassLoader;
  private TestTimeSupplier timeSupplier;

  @BeforeEach
  void setUp() throws Exception {
    reflectionCache = new ReflectionCache();
    artifactEncoding = () -> UTF_8;
    extensionClassLoader = this.getClass().getClassLoader();
    timeSupplier = new TestTimeSupplier(System.currentTimeMillis());

    // Setup basic extension model
    when(extensionModel.getName()).thenReturn(EXTENSION_NAME);
    when(extensionModel.getConfigurationModels()).thenReturn(singletonList(configurationModel));
    when(extensionModel.getConnectionProviders()).thenReturn(emptyList());

    // Setup configuration model with no parameters to avoid validation issues
    when(configurationModel.getName()).thenReturn(CONFIG_NAME);
    when(configurationModel.getConnectionProviders()).thenReturn(emptyList());
    mockParameters(configurationModel); // No parameters

    // Setup connection provider model with no parameters
    mockParameters(connectionProviderModel); // No parameters
    when(connectionProviderModel.getName()).thenReturn("testConnectionProvider");
    mockConnectionProviderFactory(connectionProviderModel);

    // Setup mule context
    when(muleContext.getConfiguration().getDynamicConfigExpiration().getExpirationPolicy())
        .thenReturn(new ImmutableExpirationPolicy(5, MINUTES, timeSupplier));
    when(muleContext.getInjector()).thenReturn(injector);
    when(muleContext.getErrorTypeRepository()).thenReturn(errorTypeRepository);
    when(muleContext.getExecutionClassLoader()).thenReturn(extensionClassLoader);
    when(muleContext.getClusterId()).thenReturn("test-cluster");

    // Setup configuration provider factory
    when(configurationProviderFactory.createStaticConfigurationProvider(any(), any(), any(), any(), any()))
        .thenReturn(staticConfigurationProvider);
    when(configurationProviderFactory.createDynamicConfigurationProvider(any(), any(), any(), any(), any(), any()))
        .thenReturn(dynamicConfigurationProvider);

    // Setup connection provider value resolver
    when(connectionProviderValueResolver.isDynamic()).thenReturn(false);
    when(connectionProviderValueResolver.getResolverSet()).thenReturn(empty());

    // Setup component parameterization to return the connection provider model
    when(componentParameterization.getModel()).thenReturn(connectionProviderModel);
    when(componentParameterization.getParameter(any(String.class), any(String.class))).thenReturn(empty());

    // Setup connection provider settings
    when(connectionProviderSettings.getConnectionProviderModel()).thenReturn(connectionProviderModel);
    when(connectionProviderSettings.getParameters()).thenReturn(componentParameterization);
    when(connectionProviderSettings.getPoolingProfile()).thenReturn(empty());
    when(connectionProviderSettings.getReconnectionConfig()).thenReturn(empty());
  }

  @Test
  void createStaticConfigurationProvider() throws Exception {
    ConfigurationProvider provider = ConfigurationCreationUtils.createConfigurationProvider(
                                                                                            extensionModel,
                                                                                            configurationModel,
                                                                                            CONFIG_NAME,
                                                                                            emptyMap(),
                                                                                            empty(),
                                                                                            empty(),
                                                                                            configurationProviderFactory,
                                                                                            expressionManager,
                                                                                            reflectionCache,
                                                                                            PARAMETERS_OWNER,
                                                                                            dslSyntaxResolver,
                                                                                            extensionClassLoader,
                                                                                            muleContext,
                                                                                            artifactEncoding);

    assertThat(provider, is(notNullValue()));
    assertThat(provider, is(instanceOf(ConfigurationProvider.class)));
    // Verify it returns the static configuration provider since no dynamic parameters were provided
    assertThat(provider, is(staticConfigurationProvider));
    // Verify the factory was called with the correct parameters
    verify(configurationProviderFactory).createStaticConfigurationProvider(
                                                                           eq(CONFIG_NAME),
                                                                           eq(extensionModel),
                                                                           eq(configurationModel),
                                                                           any(ResolverSet.class),
                                                                           any(ConnectionProviderValueResolver.class));
  }

  @Test
  void createDynamicConfigurationProviderWithExpirationPolicy() throws Exception {
    ImmutableExpirationPolicy expirationPolicy = new ImmutableExpirationPolicy(10, MINUTES, timeSupplier);

    // Make connection resolver dynamic
    when(connectionProviderValueResolver.isDynamic()).thenReturn(true);

    ConfigurationProvider provider = ConfigurationCreationUtils.createConfigurationProvider(
                                                                                            extensionModel,
                                                                                            configurationModel,
                                                                                            CONFIG_NAME,
                                                                                            emptyMap(),
                                                                                            of(expirationPolicy),
                                                                                            of(connectionProviderValueResolver),
                                                                                            configurationProviderFactory,
                                                                                            expressionManager,
                                                                                            reflectionCache,
                                                                                            PARAMETERS_OWNER,
                                                                                            dslSyntaxResolver,
                                                                                            extensionClassLoader,
                                                                                            muleContext,
                                                                                            artifactEncoding);

    assertThat(provider, is(notNullValue()));
    // Verify it returns the dynamic configuration provider since connection resolver is dynamic
    assertThat(provider, is(dynamicConfigurationProvider));
    // Verify the factory was called with the correct parameters including the custom expiration policy
    verify(configurationProviderFactory).createDynamicConfigurationProvider(
                                                                            eq(CONFIG_NAME),
                                                                            eq(extensionModel),
                                                                            eq(configurationModel),
                                                                            any(ResolverSet.class),
                                                                            eq(connectionProviderValueResolver),
                                                                            eq(expirationPolicy));
  }

  @Test
  void createConfigurationProviderWithoutConnectionProvider() throws Exception {
    // Override to ensure the extension doesn't support connectivity
    when(extensionModel.getConnectionProviders()).thenReturn(emptyList());
    when(configurationModel.getConnectionProviders()).thenReturn(emptyList());

    ConfigurationProvider provider = ConfigurationCreationUtils.createConfigurationProvider(
                                                                                            extensionModel,
                                                                                            configurationModel,
                                                                                            CONFIG_NAME,
                                                                                            emptyMap(),
                                                                                            empty(),
                                                                                            empty(),
                                                                                            configurationProviderFactory,
                                                                                            expressionManager,
                                                                                            reflectionCache,
                                                                                            PARAMETERS_OWNER,
                                                                                            dslSyntaxResolver,
                                                                                            extensionClassLoader,
                                                                                            muleContext,
                                                                                            artifactEncoding);

    assertThat(provider, is(notNullValue()));
  }

  @Test
  void createConfigurationProviderWithExplicitConnectionProvider() throws Exception {
    ConfigurationProvider provider = ConfigurationCreationUtils.createConfigurationProvider(
                                                                                            extensionModel,
                                                                                            configurationModel,
                                                                                            CONFIG_NAME,
                                                                                            emptyMap(),
                                                                                            empty(),
                                                                                            of(connectionProviderValueResolver),
                                                                                            configurationProviderFactory,
                                                                                            expressionManager,
                                                                                            reflectionCache,
                                                                                            PARAMETERS_OWNER,
                                                                                            dslSyntaxResolver,
                                                                                            extensionClassLoader,
                                                                                            muleContext,
                                                                                            artifactEncoding);

    assertThat(provider, is(notNullValue()));
    assertThat(provider, is(staticConfigurationProvider));
    // Verify the explicitly provided connection provider value resolver was used
    verify(connectionProviderValueResolver).getResolverSet();
    verify(connectionProviderValueResolver).isDynamic();
  }

  @Test
  void createConfigurationProviderWithEmptyParameters() throws Exception {
    ConfigurationProvider provider = ConfigurationCreationUtils.createConfigurationProvider(
                                                                                            extensionModel,
                                                                                            configurationModel,
                                                                                            CONFIG_NAME,
                                                                                            emptyMap(),
                                                                                            empty(),
                                                                                            empty(),
                                                                                            configurationProviderFactory,
                                                                                            expressionManager,
                                                                                            reflectionCache,
                                                                                            PARAMETERS_OWNER,
                                                                                            dslSyntaxResolver,
                                                                                            extensionClassLoader,
                                                                                            muleContext,
                                                                                            artifactEncoding);

    assertThat(provider, is(notNullValue()));
  }

  @Test
  void createConnectionProviderResolverNonOAuth() throws Exception {
    when(connectionProviderModel.getModelProperty(OAuthModelProperty.class)).thenReturn(empty());

    ConnectionProviderResolver<Object> resolver = ConfigurationCreationUtils.createConnectionProviderResolver(
                                                                                                              extensionModel,
                                                                                                              connectionProviderSettings,
                                                                                                              configurationProperties,
                                                                                                              expressionManager,
                                                                                                              reflectionCache,
                                                                                                              PARAMETERS_OWNER,
                                                                                                              dslSyntaxResolver,
                                                                                                              muleContext,
                                                                                                              artifactEncoding);

    assertThat(resolver, is(notNullValue()));
    assertThat(resolver, is(instanceOf(ConnectionProviderResolver.class)));
    // Verify that OAuth model property was checked
    verify(connectionProviderModel).getModelProperty(OAuthModelProperty.class);
    // Verify the connection provider settings were accessed
    verify(connectionProviderSettings).getConnectionProviderModel();
    verify(connectionProviderSettings).getParameters();
  }

  @Test
  void createConnectionProviderResolverWithAuthorizationCodeOAuth() throws Exception {
    AuthorizationCodeGrantType grantType = mock(AuthorizationCodeGrantType.class);
    OAuthModelProperty oauthProperty = mock(OAuthModelProperty.class);
    AuthorizationCodeOAuthHandler authCodeHandler = mock(AuthorizationCodeOAuthHandler.class);

    when(oauthProperty.getGrantTypes()).thenReturn(singletonList(grantType));
    when(connectionProviderModel.getModelProperty(OAuthModelProperty.class)).thenReturn(of(oauthProperty));
    when(connectionProviderSettings.getAuthorizationCodeOAuthHandler()).thenReturn(authCodeHandler);

    // Make the grant type call the visitor's visit method
    doAnswer(invocation -> {
      OAuthGrantTypeVisitor visitor = invocation.getArgument(0);
      visitor.visit(grantType);
      return null;
    }).when(grantType).accept(any());

    ConnectionProviderResolver<Object> resolver = ConfigurationCreationUtils.createConnectionProviderResolver(
                                                                                                              extensionModel,
                                                                                                              connectionProviderSettings,
                                                                                                              configurationProperties,
                                                                                                              expressionManager,
                                                                                                              reflectionCache,
                                                                                                              PARAMETERS_OWNER,
                                                                                                              dslSyntaxResolver,
                                                                                                              muleContext,
                                                                                                              artifactEncoding);

    assertThat(resolver, is(notNullValue()));
    assertThat(resolver, is(instanceOf(ConnectionProviderResolver.class)));
    // Verify OAuth model property was accessed (called twice: once to check, once to get grant types)
    verify(connectionProviderModel, atLeast(1)).getModelProperty(OAuthModelProperty.class);
    // Verify the authorization code OAuth handler was retrieved from settings
    verify(connectionProviderSettings).getAuthorizationCodeOAuthHandler();
    // Verify the grant type visitor pattern was used
    verify(grantType).accept(any());
  }

  @Test
  void createConnectionProviderResolverWithClientCredentialsOAuth() throws Exception {
    ClientCredentialsGrantType grantType = mock(ClientCredentialsGrantType.class);
    OAuthModelProperty oauthProperty = mock(OAuthModelProperty.class);
    ClientCredentialsOAuthHandler clientCredentialsHandler = mock(ClientCredentialsOAuthHandler.class);

    when(oauthProperty.getGrantTypes()).thenReturn(singletonList(grantType));
    when(connectionProviderModel.getModelProperty(OAuthModelProperty.class)).thenReturn(of(oauthProperty));
    when(connectionProviderSettings.getClientCredentialsOAuthHandler()).thenReturn(clientCredentialsHandler);

    // Make the grant type call the visitor's visit method
    doAnswer(invocation -> {
      OAuthGrantTypeVisitor visitor = invocation.getArgument(0);
      visitor.visit(grantType);
      return null;
    }).when(grantType).accept(any());

    ConnectionProviderResolver<Object> resolver = ConfigurationCreationUtils.createConnectionProviderResolver(
                                                                                                              extensionModel,
                                                                                                              connectionProviderSettings,
                                                                                                              configurationProperties,
                                                                                                              expressionManager,
                                                                                                              reflectionCache,
                                                                                                              PARAMETERS_OWNER,
                                                                                                              dslSyntaxResolver,
                                                                                                              muleContext,
                                                                                                              artifactEncoding);

    assertThat(resolver, is(notNullValue()));
    assertThat(resolver, is(instanceOf(ConnectionProviderResolver.class)));
    // Verify OAuth model property was accessed (called twice: once to check, once to get grant types)
    verify(connectionProviderModel, atLeast(1)).getModelProperty(OAuthModelProperty.class);
    // Verify the client credentials OAuth handler was retrieved from settings
    verify(connectionProviderSettings).getClientCredentialsOAuthHandler();
    // Verify the grant type visitor pattern was used
    verify(grantType).accept(any());
  }

  @Test
  void createConnectionProviderResolverWithPlatformManagedOAuth() throws Exception {
    PlatformManagedOAuthGrantType grantType = mock(PlatformManagedOAuthGrantType.class);
    OAuthModelProperty oauthProperty = mock(OAuthModelProperty.class);
    PlatformManagedOAuthHandler platformManagedHandler = mock(PlatformManagedOAuthHandler.class);

    when(oauthProperty.getGrantTypes()).thenReturn(singletonList(grantType));
    when(connectionProviderModel.getModelProperty(OAuthModelProperty.class)).thenReturn(of(oauthProperty));
    when(connectionProviderSettings.getPlatformManagedOAuthHandler()).thenReturn(platformManagedHandler);

    // Make the grant type call the visitor's visit method
    doAnswer(invocation -> {
      OAuthGrantTypeVisitor visitor = invocation.getArgument(0);
      visitor.visit(grantType);
      return null;
    }).when(grantType).accept(any());

    ConnectionProviderResolver<Object> resolver = ConfigurationCreationUtils.createConnectionProviderResolver(
                                                                                                              extensionModel,
                                                                                                              connectionProviderSettings,
                                                                                                              configurationProperties,
                                                                                                              expressionManager,
                                                                                                              reflectionCache,
                                                                                                              PARAMETERS_OWNER,
                                                                                                              dslSyntaxResolver,
                                                                                                              muleContext,
                                                                                                              artifactEncoding);

    assertThat(resolver, is(notNullValue()));
    assertThat(resolver, is(instanceOf(ConnectionProviderResolver.class)));
    // Verify OAuth model property was accessed (called twice: once to check, once to get grant types)
    verify(connectionProviderModel, atLeast(1)).getModelProperty(OAuthModelProperty.class);
    // Verify the platform managed OAuth handler was retrieved from settings
    verify(connectionProviderSettings).getPlatformManagedOAuthHandler();
    // Verify the grant type visitor pattern was used
    verify(grantType).accept(any());
  }

  @Test
  void createConnectionProviderResolverWithEmptyParameters() throws Exception {
    when(connectionProviderModel.getModelProperty(OAuthModelProperty.class)).thenReturn(empty());
    // componentParameterization is already setup in setUp() method

    ConnectionProviderResolver<Object> resolver = ConfigurationCreationUtils.createConnectionProviderResolver(
                                                                                                              extensionModel,
                                                                                                              connectionProviderSettings,
                                                                                                              configurationProperties,
                                                                                                              expressionManager,
                                                                                                              reflectionCache,
                                                                                                              PARAMETERS_OWNER,
                                                                                                              dslSyntaxResolver,
                                                                                                              muleContext,
                                                                                                              artifactEncoding);

    assertThat(resolver, is(notNullValue()));
    assertThat(resolver, is(instanceOf(ConnectionProviderResolver.class)));
    // Verify the componentParameterization was used to get the model
    verify(componentParameterization).getModel();
    // Verify OAuth model property was checked (should be empty)
    verify(connectionProviderModel).getModelProperty(OAuthModelProperty.class);
  }

  /**
   * Mock the connection provider factory for a connection provider model
   */
  private void mockConnectionProviderFactory(ConnectionProviderModel model) {
    ConnectionProviderFactoryModelProperty factoryProperty = mock(ConnectionProviderFactoryModelProperty.class);
    ConnectionProviderFactory<?> factory = mock(ConnectionProviderFactory.class);
    lenient().when(factoryProperty.getConnectionProviderFactory()).thenReturn(factory);
    lenient().when(model.getModelProperty(ConnectionProviderFactoryModelProperty.class)).thenReturn(of(factoryProperty));
  }

  /**
   * Test configuration class for testing purposes
   */
  public static class TestConfig {

    private String testParam;

    public String getTestParam() {
      return testParam;
    }

    public void setTestParam(String testParam) {
      this.testParam = testParam;
    }
  }
}

