/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.util;

import static org.mule.runtime.module.extension.internal.loader.utils.AnnotationsIntrospectorUtils.extractAnnotations;
import static org.mule.sdk.api.meta.Category.SELECT;
import static org.mule.sdk.api.meta.ExternalLibraryType.NATIVE;
import static org.mule.sdk.api.meta.JavaVersion.JAVA_11;
import static org.mule.sdk.api.meta.JavaVersion.JAVA_17;
import static org.mule.sdk.api.meta.JavaVersion.JAVA_8;
import static org.mule.test.allure.AllureConstants.Sdk.SDK;

import static java.util.Arrays.stream;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.aMapWithSize;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.extension.api.annotation.Operations;
import org.mule.runtime.extension.api.annotation.Sources;
import org.mule.runtime.extension.api.annotation.SubTypeMapping;
import org.mule.runtime.extension.api.annotation.connectivity.ConnectionProviders;
import org.mule.runtime.extension.api.annotation.deprecated.Deprecated;
import org.mule.runtime.extension.api.annotation.notification.NotificationActions;
import org.mule.sdk.api.annotation.Export;
import org.mule.sdk.api.annotation.Extension;
import org.mule.sdk.api.annotation.ExternalLib;
import org.mule.sdk.api.annotation.JavaVersionSupport;
import org.mule.sdk.api.annotation.OnException;
import org.mule.sdk.api.annotation.error.ErrorTypes;
import org.mule.tck.size.SmallTest;
import org.mule.test.heisenberg.extension.AsyncHeisenbergSource;
import org.mule.test.heisenberg.extension.DEARadioSource;
import org.mule.test.heisenberg.extension.HeisenbergClusterSource;
import org.mule.test.heisenberg.extension.HeisenbergConnectionProvider;
import org.mule.test.heisenberg.extension.HeisenbergErrors;
import org.mule.test.heisenberg.extension.HeisenbergExtension;
import org.mule.test.heisenberg.extension.HeisenbergNotificationAction;
import org.mule.test.heisenberg.extension.HeisenbergOperationLifecycleValidator;
import org.mule.test.heisenberg.extension.HeisenbergOperations;
import org.mule.test.heisenberg.extension.HeisenbergRouters;
import org.mule.test.heisenberg.extension.HeisenbergScopes;
import org.mule.test.heisenberg.extension.HeisenbergSource;
import org.mule.test.heisenberg.extension.HeisenbergSourceAllOptionalCallbacks;
import org.mule.test.heisenberg.extension.IgnoredHeisenbergSource;
import org.mule.test.heisenberg.extension.KillingOperations;
import org.mule.test.heisenberg.extension.MoneyLaunderingOperation;
import org.mule.test.heisenberg.extension.ReconnectableHeisenbergSdkSource;
import org.mule.test.heisenberg.extension.ReconnectableHeisenbergSource;
import org.mule.test.heisenberg.extension.SdkIgnoredHeisenbergSource;
import org.mule.test.heisenberg.extension.SecureHeisenbergConnectionProvider;
import org.mule.test.heisenberg.extension.exception.SdkHeisenbergConnectionExceptionEnricher;
import org.mule.test.heisenberg.extension.model.CarDealer;
import org.mule.test.heisenberg.extension.model.CarWash;
import org.mule.test.heisenberg.extension.model.DifferedKnockableDoor;
import org.mule.test.heisenberg.extension.model.Investment;
import org.mule.test.heisenberg.extension.model.Ricin;
import org.mule.test.heisenberg.extension.model.Weapon;
import org.mule.test.heisenberg.extension.model.drugs.Drug;
import org.mule.test.heisenberg.extension.model.drugs.Meta;

import java.lang.annotation.Annotation;
import java.util.Map;

import io.qameta.allure.Feature;
import org.junit.Test;

@SmallTest
@Feature(SDK)
public class AnnotationIntrospectorUtilsTestCase {

  @Test
  public void testAnnotationsIntrospection() {
    Class<HeisenbergTestAnnotationsExtension> testClass = HeisenbergTestAnnotationsExtension.class;
    Map<Class<? extends Annotation>, Annotation> introspectedAnnotations = extractAnnotations(testClass);

    Annotation[] expectedAnnotations = testClass.getAnnotations();
    assertThat(introspectedAnnotations, aMapWithSize(expectedAnnotations.length));
    stream(expectedAnnotations)
        .forEach(expectedAnnotation -> assertThat(introspectedAnnotations.get(expectedAnnotation.annotationType()),
                                                  is(expectedAnnotation)));
  }

  @Extension(name = HeisenbergExtension.HEISENBERG, category = SELECT)
  @JavaVersionSupport({JAVA_8, JAVA_11, JAVA_17})
  @Operations({HeisenbergOperations.class, MoneyLaunderingOperation.class,
      KillingOperations.class, HeisenbergScopes.class, HeisenbergRouters.class, HeisenbergOperationLifecycleValidator.class})
  @OnException(SdkHeisenbergConnectionExceptionEnricher.class)
  @ConnectionProviders({HeisenbergConnectionProvider.class, SecureHeisenbergConnectionProvider.class})
  @Sources({HeisenbergSource.class, HeisenbergSourceAllOptionalCallbacks.class, DEARadioSource.class,
      AsyncHeisenbergSource.class})
  @org.mule.sdk.api.annotation.Sources({ReconnectableHeisenbergSource.class, HeisenbergClusterSource.class,
      IgnoredHeisenbergSource.class, SdkIgnoredHeisenbergSource.class, ReconnectableHeisenbergSdkSource.class})
  @Export(classes = {HeisenbergExtension.class, DifferedKnockableDoor.class, HeisenbergErrors.class},
      resources = "methRecipe.json")
  @SubTypeMapping(baseType = Weapon.class, subTypes = {Ricin.class})
  @SubTypeMapping(baseType = Drug.class, subTypes = {Meta.class})
  @SubTypeMapping(baseType = Investment.class, subTypes = {CarWash.class, CarDealer.class})
  @ExternalLib(name = HeisenbergExtension.HEISENBERG_LIB_NAME, description = HeisenbergExtension.HEISENBERG_LIB_DESCRIPTION,
      nameRegexpMatcher = HeisenbergExtension.HEISENBERG_LIB_FILE_NAME,
      requiredClassName = HeisenbergExtension.HEISENBERG_LIB_CLASS_NAME, type = NATIVE,
      coordinates = "org.mule.libs:this-is-a-lib:dll:1.0.0")
  @Deprecated(message = "This extension has been deprecated because Breaking Bad has ended, use Better Call Saul extension.",
      since = "1.4.0")
  @ErrorTypes(HeisenbergErrors.class)
  @NotificationActions(HeisenbergNotificationAction.class)
  private static class HeisenbergTestAnnotationsExtension {
  }

}
