/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.http.support.internal.client;

import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.tcp.TcpClientSocketProperties;
import org.mule.runtime.http.support.internal.protocol.Http1ProtocolConfigurerImpl;
import org.mule.runtime.http.support.internal.protocol.Http2ProtocolConfigurerImpl;
import org.mule.sdk.api.http.client.HttpClientConfigurer;
import org.mule.sdk.api.http.client.proxy.ProxyConfigurer;
import org.mule.sdk.api.http.protocol.Http1ProtocolConfigurer;
import org.mule.sdk.api.http.protocol.Http2ProtocolConfigurer;
import org.mule.sdk.api.http.tcp.TcpSocketPropertiesConfigurer;

import java.util.function.Consumer;

public record HttpClientConfigToBuilder(HttpClientConfiguration.Builder builder) implements HttpClientConfigurer {

  @Override
  public HttpClientConfigurer setTlsContextFactory(TlsContextFactory tlsContextFactory) {
    builder.setTlsContextFactory(tlsContextFactory);
    return this;
  }

  @Override
  public HttpClientConfigurer setMaxConnections(int maxConnections) {
    builder.setMaxConnections(maxConnections);
    return this;
  }

  @Override
  public HttpClientConfigurer setUsePersistentConnections(boolean usePersistentConnections) {
    builder.setUsePersistentConnections(usePersistentConnections);
    return this;
  }

  @Override
  public HttpClientConfigurer setConnectionIdleTimeout(int connectionIdleTimeout) {
    builder.setConnectionIdleTimeout(connectionIdleTimeout);
    return this;
  }

  @Override
  public HttpClientConfigurer setStreaming(boolean streaming) {
    builder.setStreaming(streaming);
    return this;
  }

  @Override
  public HttpClientConfigurer setResponseBufferSize(int responseBufferSize) {
    builder.setResponseBufferSize(responseBufferSize);
    return this;
  }

  @Override
  public HttpClientConfigurer setName(String name) {
    builder.setName(name);
    return this;
  }

  @Override
  public HttpClientConfigurer setDecompress(Boolean decompress) {
    builder.setDecompress(decompress);
    return this;
  }

  @Override
  public HttpClientConfigurer configClientSocketProperties(Consumer<TcpSocketPropertiesConfigurer> configCallback) {
    var propsBuilder = TcpClientSocketProperties.builder();
    var configurer = new TcpSocketPropertiesConfigurerToBuilder(propsBuilder);
    configCallback.accept(configurer);
    builder.setClientSocketProperties(propsBuilder.build());
    return this;
  }

  @Override
  public HttpClientConfigurer configProxy(Consumer<ProxyConfigurer> configCallback) {
    var configurer = new ProxyConfigImpl();
    configCallback.accept(configurer);
    builder.setProxyConfig(configurer.build());
    return this;
  }

  @Override
  public HttpClientConfigurer configHttp1(Consumer<Http1ProtocolConfigurer> configCallback) {
    var config = new org.mule.runtime.http.api.Http1ProtocolConfig(true);
    configCallback.accept(new Http1ProtocolConfigurerImpl(config));
    builder.setHttp1Config(config);
    return this;
  }

  @Override
  public HttpClientConfigurer configHttp2(Consumer<Http2ProtocolConfigurer> configCallback) {
    var config = new org.mule.runtime.http.api.Http2ProtocolConfig(true);
    configCallback.accept(new Http2ProtocolConfigurerImpl(config));
    builder.setHttp2Config(config);
    return this;
  }
}
