/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.http.support.internal.server;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.LifecycleException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.sdk.api.http.HttpConstants;
import org.mule.sdk.api.http.server.EndpointLifecycleHandler;
import org.mule.sdk.api.http.server.HttpServer;
import org.mule.sdk.api.http.server.RequestHandler;
import org.mule.sdk.api.http.server.ServerAddress;
import org.mule.sdk.api.http.sse.server.SseClient;
import org.mule.sdk.api.http.sse.server.SseEndpointManager;
import org.mule.sdk.api.http.sse.server.SseRequestContext;

import java.io.IOException;
import java.util.Collection;
import java.util.function.Consumer;

public class HttpServerWrapper implements HttpServer {

  private final org.mule.runtime.http.api.server.HttpServer delegate;
  private final ServerAddress serverAddress;

  public HttpServerWrapper(org.mule.runtime.http.api.server.HttpServer delegate) {
    this.delegate = delegate;
    this.serverAddress = new ServerAddressWrapper(delegate.getServerAddress());
  }

  @Override
  public void start() throws MuleException {
    try {
      delegate.start();
    } catch (IOException e) {
      throw new LifecycleException(e, this);
    }
  }

  @Override
  public void stop() {
    delegate.stop();
  }

  @Override
  public void dispose() {
    delegate.dispose();
  }

  @Override
  public ServerAddress getServerAddress() {
    return serverAddress;
  }

  @Override
  public HttpConstants.Protocol getProtocol() {
    return HttpConstants.Protocol.valueOf(delegate.getProtocol().name());
  }

  @Override
  public boolean isStopping() {
    return delegate.isStopping();
  }

  @Override
  public boolean isStopped() {
    return delegate.isStopped();
  }

  @Override
  public void enableTls(TlsContextFactory tlsContextFactory) {
    delegate.enableTls(tlsContextFactory);
  }

  @Override
  public void disableTls() {
    delegate.disableTls();
  }

  @Override
  public EndpointLifecycleHandler addRequestHandler(Collection<String> methods, String path, RequestHandler requestHandler) {
    return new EndpointLifecycleHandlerWrapper(delegate.addRequestHandler(methods, path,
                                                                          new RequestHandlerWrapper(requestHandler)));
  }

  @Override
  public EndpointLifecycleHandler addRequestHandler(String path, RequestHandler requestHandler) {
    return new EndpointLifecycleHandlerWrapper(delegate.addRequestHandler(path, new RequestHandlerWrapper(requestHandler)));
  }

  @Override
  public SseEndpointManager sse(String ssePath, Consumer<SseRequestContext> onRequest, Consumer<SseClient> onClient) {
    return new SseEndpointManagerWrapper(delegate.sse(ssePath,
                                                      ctx -> onRequest.accept(new SseRequestContextWrapper(ctx)),
                                                      sseClient -> onClient.accept(new SseClientWrapper(sseClient))));
  }
}
