/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.http.support.internal.server;

import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.Http1ProtocolConfig;
import org.mule.runtime.http.api.Http2ProtocolConfig;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.support.internal.protocol.Http1ProtocolConfigurerImpl;
import org.mule.runtime.http.support.internal.protocol.Http2ProtocolConfigurerImpl;
import org.mule.sdk.api.http.protocol.Http1ProtocolConfigurer;
import org.mule.sdk.api.http.protocol.Http2ProtocolConfigurer;
import org.mule.sdk.api.http.server.HttpServerConfigurer;

import java.util.function.Consumer;
import java.util.function.Supplier;

public final class HttpServerConfigToBuilder implements HttpServerConfigurer {

  private final HttpServerConfiguration.Builder builder;

  public HttpServerConfigToBuilder(HttpServerConfiguration.Builder builder) {
    this.builder = builder;
  }

  @Override
  public HttpServerConfigurer setHost(String host) {
    builder.setHost(host);
    return this;
  }

  @Override
  public HttpServerConfigurer setPort(int port) {
    builder.setPort(port);
    return this;
  }

  @Override
  public HttpServerConfigurer setTlsContextFactory(TlsContextFactory tlsContextFactory) {
    builder.setTlsContextFactory(tlsContextFactory);
    return this;
  }

  @Override
  public HttpServerConfigurer setUsePersistentConnections(boolean usePersistentConnections) {
    builder.setUsePersistentConnections(usePersistentConnections);
    return this;
  }

  @Override
  public HttpServerConfigurer setConnectionIdleTimeout(int connectionIdleTimeout) {
    builder.setConnectionIdleTimeout(connectionIdleTimeout);
    return this;
  }

  @Override
  public HttpServerConfigurer setSchedulerSupplier(Supplier<Scheduler> schedulerSupplier) {
    builder.setSchedulerSupplier(schedulerSupplier);
    return this;
  }

  @Override
  public HttpServerConfigurer setName(String name) {
    builder.setName(name);
    return this;
  }

  @Override
  public HttpServerConfigurer setReadTimeout(long readTimeout) {
    builder.setReadTimeout(readTimeout);
    return this;
  }

  @Override
  public HttpServerConfigurer configHttp1(Consumer<Http1ProtocolConfigurer> configCallback) {
    var config = new Http1ProtocolConfig(true);
    configCallback.accept(new Http1ProtocolConfigurerImpl(config));
    builder.setHttp1Config(config);
    return this;
  }

  @Override
  public HttpServerConfigurer configHttp2(Consumer<Http2ProtocolConfigurer> configCallback) {
    var config = new Http2ProtocolConfig(true);
    configCallback.accept(new Http2ProtocolConfigurerImpl(config));
    builder.setHttp2Config(config);
    return this;
  }
}
