/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.log4j.internal;

import static org.mule.runtime.api.util.MuleSystemProperties.MULE_LOG_SEPARATION_DISABLED;
import static org.mule.runtime.api.util.MuleSystemProperties.SINGLE_APP_MODE_CONTAINER_USE_APP_LOG4J_CONFIGURATION;
import static org.mule.runtime.api.util.MuleSystemProperties.SINGLE_APP_MODE_PROPERTY;
import static org.mule.runtime.module.log4j.internal.MuleLog4jConfiguratorUtils.configureSelector;
import static org.mule.runtime.module.log4j.internal.MuleLog4jConfiguratorUtils.getDefaultReconfigurationAction;

import static java.lang.ClassLoader.getSystemClassLoader;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.IsInstanceOf.instanceOf;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.mockConstruction;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.module.log4j.boot.api.MuleLog4jContextFactory;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.util.function.Consumer;

import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.selector.ContextSelector;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junitpioneer.jupiter.SetSystemProperty;
import org.junitpioneer.jupiter.SetSystemProperty.SetSystemProperties;
import org.mockito.ArgumentCaptor;
import org.mockito.Captor;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

@ExtendWith(MockitoExtension.class)
class MuleLog4jConfiguratorUtilsTestCase extends AbstractMuleTestCase {

  @Mock
  MuleLog4jContextFactory contextFactory;

  @Captor
  ArgumentCaptor<ContextSelector> contextSelectorCapturer;
  @Captor
  ArgumentCaptor<Consumer<ContextSelector>> disposerCapturer;

  @Test
  void logSeparationWithMultipleApps() {
    configureSelector(contextFactory);

    verify(contextFactory).setContextSelector(contextSelectorCapturer.capture(), disposerCapturer.capture());

    var contextSelector = contextSelectorCapturer.getValue();
    assertThat(contextSelector, instanceOf(ArtifactAwareContextSelector.class));

    contextSelector = spy(contextSelector);
    disposerCapturer.getValue().accept(contextSelector);
    verify((Disposable) contextSelector).dispose();
  }

  @Test
  @SetSystemProperties({
      @SetSystemProperty(key = SINGLE_APP_MODE_PROPERTY, value = "true"),
      @SetSystemProperty(key = MULE_LOG_SEPARATION_DISABLED, value = "true")
  })
  void logSeparationDisabled() {
    configureSelector(contextFactory);

    verify(contextFactory).setContextSelector(contextSelectorCapturer.capture(), disposerCapturer.capture());

    var contextSelector = contextSelectorCapturer.getValue();
    assertThat(contextSelector, instanceOf(SimpleContextSelector.class));

    disposerCapturer.getValue().accept(contextSelector);
  }

  @Test
  @SetSystemProperties({
      @SetSystemProperty(key = SINGLE_APP_MODE_PROPERTY, value = "true"),
      @SetSystemProperty(key = SINGLE_APP_MODE_CONTAINER_USE_APP_LOG4J_CONFIGURATION, value = "true")
  })
  void singleApp() {
    var loggerContext = mock(LoggerContext.class);
    var appLoggerContext = mock(LoggerContext.class);
    var artifactClassLoader = new ClassLoader() {};

    try (var mockConstruction = mockConstruction(MuleLoggerContextFactory.class, (mock, context) -> {
      when(mock.build(eq(getSystemClassLoader()), any(), anyBoolean())).thenReturn(loggerContext);
      when(mock.build(eq(artifactClassLoader), any(), anyBoolean())).thenReturn(appLoggerContext);
    })) {
      configureSelector(contextFactory);

      verify(contextFactory).setContextSelector(contextSelectorCapturer.capture(), disposerCapturer.capture());

      var contextSelector = contextSelectorCapturer.getValue();
      assertThat(contextSelector, instanceOf(ApplicationReconfigurableLoggerContextSelector.class));

      disposerCapturer.getValue().accept(contextSelector);

      getDefaultReconfigurationAction().accept(artifactClassLoader);
      verify(loggerContext).reconfigure();
      verify(appLoggerContext).reconfigure();
    }
  }
}
