/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.impl.configuration;

import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_CA_FILE_LOCATION;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_CERT_FILE_LOCATION;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENDPOINT;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_COMPRESSION_TYPE;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_HEADERS;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_KEY_FILE_LOCATION;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_TIMEOUT;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_TLS_ENABLED;
import static org.mule.runtime.logging.otel.impl.configuration.ExporterBackoffConfigurator.enableBackoffStrategy;
import static org.mule.runtime.module.observability.configuration.HeadersParser.parseHeaders;

import static java.lang.Boolean.FALSE;
import static java.lang.Long.parseLong;
import static java.nio.file.Files.readAllBytes;
import static java.nio.file.Paths.get;
import static java.util.concurrent.TimeUnit.MILLISECONDS;

import static org.apache.commons.lang3.StringUtils.isEmpty;

import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfiguration;

import java.io.IOException;

import io.opentelemetry.exporter.otlp.http.logs.OtlpHttpLogRecordExporter;
import io.opentelemetry.exporter.otlp.http.logs.OtlpHttpLogRecordExporterBuilder;
import io.opentelemetry.sdk.logs.export.LogRecordExporter;

// TODO W-19294390: Generalize HttpExporterConfigurator and GRPCExporterConfigurator.
public class HttpLoggingExporterConfigurator implements LoggingExporterConfigurator {

  @Override
  public LogRecordExporter configureExporter(ObservabilitySignalConfiguration configuration)
      throws LogRecordExporterConfiguratorException {

    OtlpHttpLogRecordExporterBuilder httpLogRecordExporterBuilder = OtlpHttpLogRecordExporter.builder();

    String endpoint = configuration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENDPOINT);

    // If we have an endpoint, we set it.
    if (!isEmpty(endpoint)) {
      httpLogRecordExporterBuilder.setEndpoint(endpoint);
    }

    // We verify if we have a compression type and set it.
    String compressionType = configuration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_COMPRESSION_TYPE);
    if (compressionType != null) {
      httpLogRecordExporterBuilder.setCompression(compressionType);
    }

    // If we must enable tls, we do it.
    if (configuration.getBooleanValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_TLS_ENABLED, FALSE)) {
      configureTls(httpLogRecordExporterBuilder, configuration);
    }

    // Headers
    String headers = configuration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_HEADERS);
    if (headers != null) {
      configureHeaders(httpLogRecordExporterBuilder, headers);
    }

    // Timeout
    String timeout = configuration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_TIMEOUT);
    if (timeout != null) {
      configureTimeout(httpLogRecordExporterBuilder, timeout);
    }

    // Backoff strategy
    enableBackoffStrategy(httpLogRecordExporterBuilder, configuration);

    return httpLogRecordExporterBuilder.build();
  }

  private void configureTimeout(OtlpHttpLogRecordExporterBuilder builder, String timeout) {
    builder.setTimeout(parseLong(timeout), MILLISECONDS);
  }

  private void configureHeaders(OtlpHttpLogRecordExporterBuilder builder, String headers)
      throws LogRecordExporterConfiguratorException {
    try {
      parseHeaders(headers).forEach(builder::addHeader);
    } catch (Exception e) {
      throw new LogRecordExporterConfiguratorException(e);
    }
  }

  private void configureTls(OtlpHttpLogRecordExporterBuilder builder, ObservabilitySignalConfiguration configuration)
      throws LogRecordExporterConfiguratorException {
    configureTrustedCertificates(builder, configuration);
    configureClientTls(builder, configuration);
  }

  private static void configureTrustedCertificates(OtlpHttpLogRecordExporterBuilder builder,
                                                   ObservabilitySignalConfiguration configuration)
      throws LogRecordExporterConfiguratorException {
    String caFilePath = configuration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_CA_FILE_LOCATION);
    if (caFilePath != null) {
      try {
        byte[] caFileBytes = readAllBytes(get(caFilePath));
        builder.setTrustedCertificates(caFileBytes);
      } catch (IOException e) {
        throw new LogRecordExporterConfiguratorException(e);
      }
    }
  }

  private void configureClientTls(OtlpHttpLogRecordExporterBuilder builder, ObservabilitySignalConfiguration configuration)
      throws LogRecordExporterConfiguratorException {
    String certFilePath = configuration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_CERT_FILE_LOCATION);
    String keyFilePath = configuration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_KEY_FILE_LOCATION);

    if (certFilePath != null && keyFilePath != null) {
      try {
        byte[] keyFileBytes = readAllBytes(get(keyFilePath));
        byte[] certFileBytes = readAllBytes(get(certFilePath));
        builder.setClientTls(keyFileBytes, certFileBytes);
      } catch (IOException e) {
        throw new LogRecordExporterConfiguratorException(e);
      }
    }
  }
}
