/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.impl.configuration;

import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BACKOFF_MULTIPLIER;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_INITIAL_BACKOFF;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_MAX_BACKOFF;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BACKOFF_MAX_ATTEMPTS;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfiguration;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import io.opentelemetry.sdk.common.export.RetryPolicy;

public class ExporterBackoffConfigurator {

  /**
   * <p>
   * Enables the experimental backoff strategy for the Open Telemetry exporters.
   * </p>
   * <p>
   * A default parametrization will be used if no configuration is provided:
   * </p>
   * <lu>
   * <li>Initial backoff: 1 Second</li>
   * <li>Maximum backoff: 5 Seconds</li>
   * <li>Maximum retry attempts: 5</li>
   * <li>Backoff multiplier: 1.5</li> </lu>
   *
   * @param exporterBuilder The exporter builder.
   * @param configuration   The configuration that will be used to override the default backoff parameters.
   */
  public static void enableBackoffStrategy(Object exporterBuilder, ObservabilitySignalConfiguration configuration) {
    try {
      // Since it's an experimental feature, we must enable it using reflection.
      // See https://github.com/open-telemetry/opentelemetry-java/pull/3791
      Field delegateField = exporterBuilder.getClass().getDeclaredField("delegate");
      delegateField.setAccessible(true);
      Method setRetryPolicyMethod =
          delegateField.get(exporterBuilder).getClass().getDeclaredMethod("setRetryPolicy", RetryPolicy.class);
      setRetryPolicyMethod.setAccessible(true);
      setRetryPolicyMethod.invoke(delegateField.get(exporterBuilder), new RetryPolicyBuilder(configuration).build());
    } catch (NoSuchFieldException | IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
      throw new MuleRuntimeException(new IllegalArgumentException("Unable to set the RetryPolicy reflectively.", e));
    }
  }

  /**
   * Builder that encapsulates the configuration of the Open Telemetry {@link RetryPolicy}.
   */
  private record RetryPolicyBuilder(ObservabilitySignalConfiguration configuration) {

    // TODO: W-19283907: Fully declarative configuration instead of picking up properties from the config.
    public RetryPolicy build() {
      return RetryPolicy.builder()
          .setBackoffMultiplier(configuration.getDoubleValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BACKOFF_MULTIPLIER))
          .setInitialBackoff(configuration.getSecondsDurationValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_INITIAL_BACKOFF))
          .setMaxBackoff(configuration.getSecondsDurationValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_MAX_BACKOFF))
          .setMaxAttempts(configuration.getIntValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BACKOFF_MAX_ATTEMPTS))
          .build();
    }
  }
}
