/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.impl.export.sniffer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.logs.data.LogRecordData;
import io.opentelemetry.sdk.logs.export.LogRecordExporter;

/**
 * A LogRecord exporter that will capture and return all the exported LogRecord instances for testing purposes.
 *
 * @see LogRecordData
 * @since 4.10
 */
public class SniffedLogRecordExporter implements LogRecordExporter {

  private final LogRecordExporter delegate;
  private final List<InternalExportedLogRecordSniffer> exportedLogRecordSniffers = new ArrayList<>(1);

  public SniffedLogRecordExporter(LogRecordExporter delegate) {
    this.delegate = delegate;
  }

  @Override
  public CompletableResultCode export(Collection<LogRecordData> logs) {
    exportedLogRecordSniffers.forEach(exportedLogRecordSniffer -> exportedLogRecordSniffer.sniff(logs));
    return delegate.export(logs);
  }

  @Override
  public CompletableResultCode flush() {
    return delegate.flush();
  }

  @Override
  public CompletableResultCode shutdown() {
    return delegate.shutdown();
  }

  @Override
  public void close() {
    LogRecordExporter.super.close();
    delegate.close();
  }

  public ExportedLogRecordSniffer getExportedLogRecordSniffer() {
    InternalExportedLogRecordSniffer exportedLogRecordSniffer = new InternalExportedLogRecordSniffer();
    exportedLogRecordSniffers.add(exportedLogRecordSniffer);
    return exportedLogRecordSniffer;
  }

  private static class InternalExportedLogRecordSniffer implements ExportedLogRecordSniffer {

    private final List<SniffedLogRecord> sniffedLogRecords = new CopyOnWriteArrayList<>();

    public void sniff(Collection<LogRecordData> sniffedLogRecord) {
      sniffedLogRecord.stream().map(InternalSniffedLogRecord::new).forEach(sniffedLogRecords::add);
    }

    @Override
    public List<SniffedLogRecord> getSniffedLogRecords() {
      return sniffedLogRecords;
    }
  }

  private static class InternalSniffedLogRecord implements SniffedLogRecord {

    LogRecordData sniffedLogRecord;

    public InternalSniffedLogRecord(LogRecordData sniffedLogRecord) {
      this.sniffedLogRecord = sniffedLogRecord;
    }

    @Override
    public String getBody() {
      return sniffedLogRecord.getBodyValue() != null ? sniffedLogRecord.getBodyValue().asString() : "";
    }

    @Override
    public String getSpanId() {
      return sniffedLogRecord.getSpanContext().getSpanId();
    }

    @Override
    public String getTraceId() {
      return sniffedLogRecord.getSpanContext().getTraceId();
    }

    @Override
    public String getSeverity() {
      return sniffedLogRecord.getSeverity().toString();
    }
  }
}
