/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.impl.export.log4j;

import java.io.Serializable;
import java.util.function.Supplier;

import io.opentelemetry.instrumentation.log4j.appender.v2_17.OpenTelemetryAppender;
import io.opentelemetry.sdk.OpenTelemetrySdk;
import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.ErrorHandler;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;

/**
 * Open Telemetry Appender decorator that allows lifecycle extension plus OpenTelemetrySdk lifecycle handling.
 */
public class DedicatedSdkOpenTelemetryAppender implements Appender {

  private final OpenTelemetryAppender delegate;

  private final Supplier<OpenTelemetrySdk> openTelemetryLoggingSdkSupplier;

  private OpenTelemetrySdk openTelemetrySdk;

  private final Runnable onInitializeHook;
  private final Runnable onStartHook;
  private final Runnable onStopHook;

  public static ExtensibleLifeCycleAppenderBuilder builder(OpenTelemetryAppender delegate) {
    return new ExtensibleLifeCycleAppenderBuilder(delegate);
  }

  private DedicatedSdkOpenTelemetryAppender(OpenTelemetryAppender delegate,
                                            Supplier<OpenTelemetrySdk> openTelemetryLoggingSdkSupplier,
                                            Runnable onInitializeHook, Runnable onStartHook, Runnable onStopHook) {
    this.delegate = delegate;
    this.openTelemetryLoggingSdkSupplier = openTelemetryLoggingSdkSupplier;
    this.onInitializeHook = onInitializeHook;
    this.onStartHook = onStartHook;
    this.onStopHook = onStopHook;
  }

  @Override
  public void append(LogEvent event) {
    delegate.append(event);
  }

  @Override
  public String getName() {
    return delegate.getName();
  }

  @Override
  public Layout<? extends Serializable> getLayout() {
    return delegate.getLayout();
  }

  @Override
  public boolean ignoreExceptions() {
    return delegate.ignoreExceptions();
  }

  @Override
  public ErrorHandler getHandler() {
    return delegate.getHandler();
  }

  @Override
  public void setHandler(ErrorHandler handler) {
    delegate.setHandler(handler);
  }

  @Override
  public State getState() {
    return delegate.getState();
  }

  @Override
  public void initialize() {
    delegate.initialize();
    onInitializeHook.run();
  }

  @Override
  public void start() {
    synchronized (this) {
      if (openTelemetrySdk != null) {
        openTelemetrySdk.close();
      }
      openTelemetrySdk = openTelemetryLoggingSdkSupplier.get();
      delegate.setOpenTelemetry(openTelemetrySdk);
      delegate.start();
      onStartHook.run();
    }
  }

  @Override
  public void stop() {
    synchronized (this) {
      if (openTelemetrySdk != null) {
        openTelemetrySdk.close();
      }
      delegate.stop();
      onStopHook.run();
    }
  }

  @Override
  public boolean isStarted() {
    return delegate.isStarted();
  }

  @Override
  public boolean isStopped() {
    return delegate.isStopped();
  }

  public static class ExtensibleLifeCycleAppenderBuilder {

    private final OpenTelemetryAppender delegate;

    private Supplier<OpenTelemetrySdk> openTelemetryLoggingSdkSupplier;

    private Runnable onInitializeHook = () -> {
    };
    private Runnable onStartHook = () -> {
    };
    private Runnable onStopHook = () -> {
    };

    private ExtensibleLifeCycleAppenderBuilder(OpenTelemetryAppender delegate) {
      this.delegate = delegate;
    }

    public ExtensibleLifeCycleAppenderBuilder withOpenTelemetryLoggingSdkProvider(Supplier<OpenTelemetrySdk> openTelemetryLoggingSdkSupplier) {
      this.openTelemetryLoggingSdkSupplier = openTelemetryLoggingSdkSupplier;
      return this;
    }

    public ExtensibleLifeCycleAppenderBuilder withOnInitializeHook(Runnable onInitializeHook) {
      this.onInitializeHook = onInitializeHook;
      return this;
    }

    public ExtensibleLifeCycleAppenderBuilder withOnStartHook(Runnable onStartHook) {
      this.onStartHook = onStartHook;
      return this;
    }

    public ExtensibleLifeCycleAppenderBuilder withOnStopHook(Runnable onStopHook) {
      this.onStopHook = onStopHook;
      return this;
    }

    public Appender build() {
      return new DedicatedSdkOpenTelemetryAppender(delegate, openTelemetryLoggingSdkSupplier, onInitializeHook,
                                                   onStartHook, onStopHook);
    }
  }
}
