/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.impl;

import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BATCH_BACKPRESSURE_STRATEGY;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BATCH_QUEUE_SIZE;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_MAX_BATCH_SIZE;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAME;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAMESPACE;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_TYPE;
import static org.mule.runtime.logging.otel.impl.configuration.OpenTelemetryAutoConfigurableLoggingConfiguration.DEFAULT_BATCH_SCHEDULE_DELAY_MILLIS;
import static org.mule.runtime.logging.otel.impl.export.batch.BlockingBatchLogRecordProcessor.builder;

import static java.lang.Boolean.getBoolean;
import static java.lang.Long.getLong;
import static java.util.concurrent.TimeUnit.MILLISECONDS;

import static io.opentelemetry.api.common.AttributeKey.stringKey;
import static io.opentelemetry.api.common.Attributes.of;
import static io.opentelemetry.sdk.logs.export.SimpleLogRecordProcessor.create;
import static io.opentelemetry.sdk.resources.Resource.create;

import org.mule.runtime.logging.otel.impl.configuration.FileOpenTelemetryLoggingConfiguration;
import org.mule.runtime.logging.otel.impl.configuration.LogRecordExporterConfiguratorException;
import org.mule.runtime.logging.otel.impl.configuration.OpenTelemetryAutoConfigurableLoggingConfiguration;
import org.mule.runtime.logging.otel.impl.configuration.OpenTelemetryLoggingExporterBackpressureStrategy;
import org.mule.runtime.logging.otel.impl.configuration.OpenTelemetryLoggingExporterTransport;
import org.mule.runtime.logging.otel.impl.export.batch.BlockingBatchLogRecordProcessorBuilder;
import org.mule.runtime.logging.otel.impl.export.batch.BlockingBatchLogRecordProcessor;
import org.mule.runtime.logging.otel.impl.export.sniffer.ExportedLogRecordSniffer;
import org.mule.runtime.logging.otel.impl.export.sniffer.SniffedLogRecordExporter;
import org.mule.runtime.module.artifact.api.descriptor.DeployableArtifactDescriptor;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfigurationFileFinder;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfiguration;

import io.opentelemetry.sdk.logs.LogRecordProcessor;
import io.opentelemetry.sdk.logs.SdkLoggerProvider;
import io.opentelemetry.sdk.logs.export.LogRecordExporter;
import io.opentelemetry.sdk.OpenTelemetrySdk;
import io.opentelemetry.sdk.resources.Resource;

/**
 * Factory that encapsulates the obtention of an open telemetry SDK capable of export artifact/container logs.
 *
 * @since 4.10.0
 */
public class OpenTelemetryLoggingSDKFactory {

  private static final String SERVICE_NAMESPACE = "service.namespace";
  private static final String SERVICE_NAME = "service.name";

  // This default constant is included here because it depends on the configuration's owner being the container.
  private static final String DEFAULT_CONTAINER_SERVICE_NAME = "mule-container";

  // Private properties only meant to be set for testing purposes.
  private static final String USE_EXPORTER_SNIFFER = MULE_OPEN_TELEMETRY_LOGGING_EXPORTER + ".use.sniffer";
  private static final String MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BATCH_SCHEDULED_DELAY =
      MULE_OPEN_TELEMETRY_LOGGING_EXPORTER + ".batch.scheduledDelay";

  private static final SniffedLogRecordExporter sniffedLogRecordExporter =
      new SniffedLogRecordExporter(LogRecordExporter.composite());

  private OpenTelemetryLoggingSDKFactory() {}

  /**
   * Creates an open telemetry SDK capable of exporting log records generated by either a deployed artifact or the mule container.
   *
   * @param deployableArtifactDescriptor The descriptor of the artifact that owns the logger context. Can be null, meaning that
   *                                     the logger context belongs to the mule container.
   * @return A logging SDK that will export log records through configured exporter.
   * @throws LogRecordExporterConfiguratorException If the configuration is invalid.
   */
  public static OpenTelemetrySdk getOpenTelemetryLoggingSDK(DeployableArtifactDescriptor deployableArtifactDescriptor,
                                                            ObservabilitySignalConfigurationFileFinder loggingSignalConfigurationFinder)
      throws LogRecordExporterConfiguratorException {
    ObservabilitySignalConfiguration loggingExportConfiguration =
        getLoggingExportConfiguration(deployableArtifactDescriptor, loggingSignalConfigurationFinder);
    // In order to maintain correlation, the service name must be the same as the rest of the artifact's telemetry signals
    // (metrics / traces).
    Resource resource = configureLoggingResource(deployableArtifactDescriptor, loggingExportConfiguration);
    SdkLoggerProvider sdkLoggerProvider = SdkLoggerProvider.builder()
        .addResource(resource)
        .addLogRecordProcessor(getLogRecordProcessor(loggingExportConfiguration)).build();
    return OpenTelemetrySdk.builder().setLoggerProvider(sdkLoggerProvider).build();
  }

  /**
   * @return A log record processor capable of exporting log records to the configured exporter.
   * @throws LogRecordExporterConfiguratorException If the configuration is invalid.
   */
  private static LogRecordProcessor getLogRecordProcessor(ObservabilitySignalConfiguration loggingExportConfiguration)
      throws LogRecordExporterConfiguratorException {
    if (getBoolean(USE_EXPORTER_SNIFFER)) {
      return create(sniffedLogRecordExporter);
    } else {
      return createBatchLogRecordProcessor(getLogRecordExporter(loggingExportConfiguration), loggingExportConfiguration);
    }
  }

  /**
   * Creates a batch log record processor according to the configured parameters.
   *
   * @param logRecordExporter          The already configured log record exporter.
   * @param loggingExportConfiguration The logging export configuration.
   * @return A batch log record processor that will send the log records to the given exporter in batches.
   * @throws LogRecordExporterConfiguratorException If the configuration is invalid.
   */
  private static BlockingBatchLogRecordProcessor createBatchLogRecordProcessor(LogRecordExporter logRecordExporter,
                                                                               ObservabilitySignalConfiguration loggingExportConfiguration)
      throws LogRecordExporterConfiguratorException {
    BlockingBatchLogRecordProcessorBuilder batchLogRecordProcessorBuilder = builder(logRecordExporter);
    int maxBatchSize = loggingExportConfiguration.getIntValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_MAX_BATCH_SIZE);
    if (maxBatchSize < 512) {
      throw new LogRecordExporterConfiguratorException("The batch max size cannot be lower than 512");
    }
    int batchQueueSize = loggingExportConfiguration.getIntValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BATCH_QUEUE_SIZE);
    return batchLogRecordProcessorBuilder
        .setMaxQueueSize(batchQueueSize)
        // TODO: W-19261036 - Need definition about what to do with this metrics (ingestion is unsure about how to consume this).
        // .setMeterProvider(getMeterProvider(loggingExportConfiguration))

        // Given to the static nature of logging configuration, we cannot use a privileged configuration approach such as the
        // tracing feature.
        // We use a plain, not API system property instead.
        .setScheduleDelay(getLong(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BATCH_SCHEDULED_DELAY,
                                  DEFAULT_BATCH_SCHEDULE_DELAY_MILLIS),
                          MILLISECONDS)
        // TODO: W-19283907: Fully declarative configuration instead of picking up properties from the config.
        .setBackpressureStrategy(OpenTelemetryLoggingExporterBackpressureStrategy
            .valueOf(loggingExportConfiguration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BATCH_BACKPRESSURE_STRATEGY)))
        .setMaxExportBatchSize(maxBatchSize).build();
  }

  /**
   * Creates a log record exporter according to the configured parameters.
   *
   * @return A log record exporter that will export the log records to the configured endpoint.
   * @throws LogRecordExporterConfiguratorException If the configuration is invalid.
   */
  private static LogRecordExporter getLogRecordExporter(ObservabilitySignalConfiguration loggingExportConfiguration)
      throws LogRecordExporterConfiguratorException {
    String type = loggingExportConfiguration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_TYPE);
    if (type == null) {
      throw new LogRecordExporterConfiguratorException("A type for log export was not configured.");
    }
    try {
      return OpenTelemetryLoggingExporterTransport.valueOf(type).getLoggingExporterConfigurator()
          .configureExporter(loggingExportConfiguration);
    } catch (Exception e) {
      throw new LogRecordExporterConfiguratorException(e);
    }
  }

  /**
   * @return A record exporter that will capture every exported record for testing / debug purposes.
   */
  public static ExportedLogRecordSniffer getLogRecordSniffer() {
    return sniffedLogRecordExporter.getExportedLogRecordSniffer();
  }

  public static ObservabilitySignalConfiguration getLoggingExportConfiguration(DeployableArtifactDescriptor deployableArtifactDescriptor,
                                                                               ObservabilitySignalConfigurationFileFinder observabilitySignalArtifactResourceFinder) {
    return new OpenTelemetryAutoConfigurableLoggingConfiguration(new FileOpenTelemetryLoggingConfiguration(observabilitySignalArtifactResourceFinder,
                                                                                                           deployableArtifactDescriptor));
  }

  /**
   * Instantiates and configures the Open Telemetry resource that will identify the source of the logging export.
   *
   * @param deployableArtifactDescriptor The artifact descriptor for which to create the logging SDK. Can be null, meaning that it
   *                                     is the mule container.
   * @param loggingExportConfiguration   The logging export configuration.
   * @return A configured Open Telemetry resource instance that includes the service.name and service.namespace attributes.
   */
  private static Resource configureLoggingResource(DeployableArtifactDescriptor deployableArtifactDescriptor,
                                                   ObservabilitySignalConfiguration loggingExportConfiguration) {
    return create(of(
                     stringKey(SERVICE_NAME), resolveServiceName(deployableArtifactDescriptor, loggingExportConfiguration),
                     stringKey(SERVICE_NAMESPACE), resolveServiceNamespace(loggingExportConfiguration)));
  }

  private static String resolveServiceName(DeployableArtifactDescriptor deployableArtifactDescriptor,
                                           ObservabilitySignalConfiguration loggingExportConfiguration) {
    if (deployableArtifactDescriptor != null) {
      return loggingExportConfiguration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAME,
                                                       deployableArtifactDescriptor.getName());
    } else {
      return loggingExportConfiguration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAME,
                                                       DEFAULT_CONTAINER_SERVICE_NAME);
    }
  }

  private static String resolveServiceNamespace(ObservabilitySignalConfiguration loggingExportConfiguration) {
    return loggingExportConfiguration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAMESPACE);
  }

}
