/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.impl.configuration;

import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.disposeIfNeeded;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BACKOFF_MAX_ATTEMPTS;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BACKOFF_MULTIPLIER;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BATCH_BACKPRESSURE_STRATEGY;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BATCH_QUEUE_SIZE;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENDPOINT;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_INITIAL_BACKOFF;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_MAX_BACKOFF;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_MAX_BATCH_SIZE;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_LEVEL;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAMESPACE;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_TIMEOUT;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_TYPE;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.ast.api.exception.PropertyNotFoundException;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfiguration;

import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;

public class OpenTelemetryAutoConfigurableLoggingConfiguration implements ObservabilitySignalConfiguration, Disposable {

  private static final Logger LOGGER = getLogger(OpenTelemetryAutoConfigurableLoggingConfiguration.class);

  public static final String HTTP_EXPORTER_TYPE = "HTTP";
  public static final String GRPC_EXPORTER_TYPE = "GRPC";
  public static final String DEFAULT_GRPC_EXPORTER_ENDPOINT = "http://localhost:4317";
  public static final String DEFAULT_HTTP_EXPORTER_ENDPOINT = "http://localhost:4318/v1/logs";

  public static final String DEFAULT_BACKOFF_MULTIPLIER = "1.5";
  public static final String DEFAULT_INITIAL_BACKOFF = "1";
  public static final String DEFAULT_MAXIMUM_BACKOFF = "5";
  public static final String DEFAULT_BACKOFF_MAX_ATTEMPTS = "5";
  public static final String DEFAULT_EXPORTER_TYPE = HTTP_EXPORTER_TYPE;
  public static final String DEFAULT_EXPORTER_TIMEOUT = "10000";

  public static final Long DEFAULT_BATCH_SCHEDULE_DELAY_MILLIS = 1000L;
  public static final String DEFAULT_BATCH_QUEUE_SIZE = "2048";
  public static final String DEFAULT_BATCH_MAX_BATCH_SIZE = "512";

  public static final String DEFAULT_SERVICE_NAMESPACE = "mule";

  public static final String DEFAULT_BATCH_BACKPRESSURE_STRATEGY = "BLOCK";

  public static final String DEFAULT_EXPORTER_LEVEL = "INFO";

  private final ObservabilitySignalConfiguration delegate;
  private final Map<String, String> defaultConfigurationValues = new HashMap<>();

  public OpenTelemetryAutoConfigurableLoggingConfiguration(ObservabilitySignalConfiguration delegate) {
    this.delegate = delegate;
    initializeConfigurationValues();
  }

  @Override
  public String getStringValue(String key) {
    try {
      return delegate.getStringValue(key, defaultConfigurationValues.get(key));
    } catch (PropertyNotFoundException e) {
      return defaultConfigurationValues.get(key);
    }
  }

  private void initializeConfigurationValues() {
    defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED, "false");
    defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_TYPE, HTTP_EXPORTER_TYPE);
    if (getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_TYPE, DEFAULT_EXPORTER_TYPE).equals(GRPC_EXPORTER_TYPE)) {
      defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENDPOINT, DEFAULT_GRPC_EXPORTER_ENDPOINT);
    } else {
      defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENDPOINT, DEFAULT_HTTP_EXPORTER_ENDPOINT);
    }
    defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_TIMEOUT, DEFAULT_EXPORTER_TIMEOUT);
    defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BACKOFF_MULTIPLIER, DEFAULT_BACKOFF_MULTIPLIER);
    defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_INITIAL_BACKOFF, DEFAULT_INITIAL_BACKOFF);
    defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_MAX_BACKOFF, DEFAULT_MAXIMUM_BACKOFF);
    defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BACKOFF_MAX_ATTEMPTS, DEFAULT_BACKOFF_MAX_ATTEMPTS);

    defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BATCH_QUEUE_SIZE, DEFAULT_BATCH_QUEUE_SIZE);
    defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_MAX_BATCH_SIZE, DEFAULT_BATCH_MAX_BATCH_SIZE);
    defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_BATCH_BACKPRESSURE_STRATEGY,
                                   DEFAULT_BATCH_BACKPRESSURE_STRATEGY);

    defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAMESPACE, DEFAULT_SERVICE_NAMESPACE);
    defaultConfigurationValues.put(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_LEVEL, DEFAULT_EXPORTER_LEVEL);
  }

  @Override
  public void dispose() {
    if (delegate != null) {
      disposeIfNeeded(delegate, LOGGER);
    }
  }
}
