/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.configuration.impl.log4j;

import static org.mule.runtime.logging.otel.impl.export.log4j.OpenTelemetryLog4JBridge.addOpenTelemetryLogging;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_CONFIGURATION_FILE_NAME;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_LEVEL;
import static org.mule.runtime.logging.otel.impl.export.log4j.OpenTelemetryLog4JBridge.OPEN_TELEMETRY_APPENDER_NAME_SUFFIX;

import static java.lang.Boolean.parseBoolean;
import static java.lang.System.clearProperty;
import static java.lang.System.getProperty;
import static java.lang.System.setProperty;
import static java.util.Collections.emptyMap;

import static org.apache.logging.log4j.Level.DEBUG;
import static org.apache.logging.log4j.Level.ERROR;
import static org.apache.logging.log4j.Level.FATAL;
import static org.apache.logging.log4j.Level.INFO;
import static org.apache.logging.log4j.Level.TRACE;
import static org.apache.logging.log4j.Level.WARN;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.jupiter.api.Assumptions.assumeTrue;
import static org.mockito.ArgumentCaptor.forClass;
import static org.mockito.ArgumentMatchers.isNull;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.util.MuleSystemProperties;
import org.mule.runtime.logging.otel.impl.configuration.LogRecordExporterConfiguratorException;
import org.mule.runtime.module.artifact.api.descriptor.DeployableArtifactDescriptor;
import org.mule.runtime.utils.parameters.SystemProperty;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.config.AbstractConfiguration;
import org.apache.logging.log4j.core.config.LoggerConfig;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.junit.jupiter.params.ParameterizedClass;
import org.junit.jupiter.params.provider.CsvSource;
import org.mockito.ArgumentCaptor;

@ParameterizedClass(name = "[{index}] {arguments}")
@CsvSource(useHeadersInDisplayName = true, value = {
    "enableOTELLogging, loggingConfigurationFile",
    "false, null",
    "true, null"
})
class OpenTelemetryLog4JBridgeTestCase {

  private final LoggerContext loggerContextStub = LoggerContext.getContext(false);

  @RegisterExtension
  public final SystemProperty enableOTELLogging;

  @RegisterExtension
  public final SystemProperty loggingConfigurationFile;

  public DeployableArtifactDescriptor artifactDescriptor;

  @RegisterExtension
  public final SystemProperty enableConfigurationAtApplicationLevel =
      new SystemProperty(MuleSystemProperties.ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY, "true");

  public OpenTelemetryLog4JBridgeTestCase(String enableOTELLogging, String loggingConfigurationFile) {
    this.enableOTELLogging = new SystemProperty(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED, enableOTELLogging);
    this.loggingConfigurationFile =
        new SystemProperty(MULE_OPEN_TELEMETRY_LOGGING_CONFIGURATION_FILE_NAME, loggingConfigurationFile);
  }

  @BeforeEach
  void setUp() {
    artifactDescriptor = mock(DeployableArtifactDescriptor.class);
    when(artifactDescriptor.getName()).thenReturn("test-artifact");
  }

  @Test
  void testLog4jBridgeWithDefaultConfiguration() throws LogRecordExporterConfiguratorException {
    addOpenTelemetryLogging(loggerContextStub, artifactDescriptor);
    Appender openTelemetryAppender = loggerContextStub.getRootLogger().getAppenders()
        .get("test-artifact" + OPEN_TELEMETRY_APPENDER_NAME_SUFFIX);
    if (parseBoolean(enableOTELLogging.getPropertyValue())) {
      assertThat(openTelemetryAppender, notNullValue());
    } else {
      assertThat(openTelemetryAppender, nullValue());
    }
  }

  @Test
  void testLog4jBridgeWithDefaultLevel() throws LogRecordExporterConfiguratorException {
    assumeTrue(parseBoolean(enableOTELLogging.getPropertyValue()),
               "Skipping test: OTEL logging is disabled");
    SpyContext spyContext = createSpyContext(loggerContextStub);
    addOpenTelemetryLogging(spyContext.context, artifactDescriptor);
    verifyAppenderLevel(spyContext.rootLogger, INFO);
  }

  @Test
  void testLog4jBridgeWithConfiguredLevel() {
    assumeTrue(parseBoolean(enableOTELLogging.getPropertyValue()),
               "Skipping test: OTEL logging is disabled");
    withSystemProperty(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_LEVEL, "WARN", () -> {
      SpyContext spyContext = createSpyContext(loggerContextStub);
      addOpenTelemetryLogging(spyContext.context, artifactDescriptor);
      verifyAppenderLevel(spyContext.rootLogger, WARN);
    });
  }

  @Test
  void testLog4jBridgeWithDifferentLevels() {
    assumeTrue(parseBoolean(enableOTELLogging.getPropertyValue()),
               "Skipping test: OTEL logging is disabled");
    String[] levelStrings = {"TRACE", "DEBUG", "INFO", "WARN", "ERROR", "FATAL"};
    Level[] expectedLevels = {TRACE, DEBUG, INFO, WARN, ERROR, FATAL};
    for (int i = 0; i < levelStrings.length; i++) {
      String levelString = levelStrings[i];
      Level expectedLevel = expectedLevels[i];
      withSystemProperty(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_LEVEL, levelString, () -> {
        LoggerContext testContext = LoggerContext.getContext(false);
        SpyContext spyContext = createSpyContext(testContext);
        addOpenTelemetryLogging(spyContext.context, artifactDescriptor);
        verifyAppenderLevel(spyContext.rootLogger, expectedLevel);
      });
    }
  }

  @Test
  void testLog4jBridgeWithExtendedLevels() {
    assumeTrue(parseBoolean(enableOTELLogging.getPropertyValue()),
               "Skipping test: OTEL logging is disabled");
    // Test that extended OpenTelemetry severity levels (INFO2, etc.) are normalized to base levels
    String[] extendedLevels = {"TRACE2", "DEBUG2", "INFO2", "WARN2", "ERROR2", "FATAL2"};
    Level[] expectedBaseLevels = {TRACE, DEBUG, INFO, WARN, ERROR, FATAL};
    for (int i = 0; i < extendedLevels.length; i++) {
      String extendedLevel = extendedLevels[i];
      Level expectedBaseLevel = expectedBaseLevels[i];
      withSystemProperty(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_LEVEL, extendedLevel, () -> {
        LoggerContext testContext = LoggerContext.getContext(false);
        SpyContext spyContext = createSpyContext(testContext);
        addOpenTelemetryLogging(spyContext.context, artifactDescriptor);
        verifyAppenderLevel(spyContext.rootLogger, expectedBaseLevel);
      });
    }
  }

  private record SpyContext(LoggerContext context, AbstractConfiguration configuration, LoggerConfig rootLogger) {

  }

  private SpyContext createSpyContext(LoggerContext loggerContext) {
    LoggerContext spyContext = spy(loggerContext);
    AbstractConfiguration spyConfiguration = spy((AbstractConfiguration) loggerContext.getConfiguration());
    LoggerConfig spyRootLogger = spy(loggerContext.getConfiguration().getRootLogger());

    when(spyContext.getConfiguration()).thenReturn(spyConfiguration);
    when(spyConfiguration.getRootLogger()).thenReturn(spyRootLogger);
    when(spyConfiguration.getLoggers()).thenReturn(emptyMap());

    return new SpyContext(spyContext, spyConfiguration, spyRootLogger);
  }

  private void verifyAppenderLevel(LoggerConfig spyRootLogger, Level expectedLevel) {
    ArgumentCaptor<Appender> appenderCaptor = forClass(Appender.class);
    ArgumentCaptor<Level> levelCaptor = forClass(Level.class);
    verify(spyRootLogger).addAppender(appenderCaptor.capture(), levelCaptor.capture(), isNull());

    assertThat(levelCaptor.getValue(), equalTo(expectedLevel));
    assertThat(appenderCaptor.getValue(), notNullValue());
  }

  private void withSystemProperty(String propertyName, String value, Runnable action) {
    String previousValue = getProperty(propertyName);
    try {
      if (value != null) {
        setProperty(propertyName, value);
      } else {
        clearProperty(propertyName);
      }
      action.run();
    } finally {
      if (previousValue == null) {
        clearProperty(propertyName);
      } else {
        setProperty(propertyName, previousValue);
      }
    }
  }

}
