/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.observability.configuration;

import java.io.File;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A watcher for changes in the configuration file.
 */
public class ObservabilityConfigurationFileWatcher extends Thread {

  public static final String MULE_OBSERVABILITY_CONFIGURATION_WATCHER_DEFAULT_DELAY_PROPERTY =
      "mule.observability.configuration.configurationFileWatcherDelay";

  private static final Logger LOGGER = LoggerFactory.getLogger(ObservabilityConfigurationFileWatcher.class);
  public final long DEFAULT_DELAY = Long
      .getLong(MULE_OBSERVABILITY_CONFIGURATION_WATCHER_DEFAULT_DELAY_PROPERTY,
               60000L);
  private final String filename;
  private final Runnable doOnChange;

  protected long delay = DEFAULT_DELAY;
  private final File file;
  private long lastModified;
  private boolean warnedAlready;

  public ObservabilityConfigurationFileWatcher(File file, Runnable doOnChange) {
    super("FileSpanExporterConfigurationWatcher");
    this.filename = file.getName();
    this.file = file;
    this.doOnChange = doOnChange;
    this.lastModified = file.lastModified();
    this.setDaemon(true);
  }

  protected void checkAndConfigure() {
    boolean fileExists;
    try {
      fileExists = file.exists();
    } catch (SecurityException var4) {
      LOGGER.warn("The tracing config file {} was possibly removed.", filename);
      this.interrupt();
      return;
    }

    if (fileExists) {
      long fileLastMod = file.lastModified();
      if (fileLastMod > lastModified) {
        this.lastModified = fileLastMod;
        this.doOnChange();
        this.warnedAlready = false;
      }
    } else if (!this.warnedAlready) {
      LOGGER.warn("Configuration for file exporter was not found. It was possibly removed.");
      this.warnedAlready = true;
    }
  }

  @Override
  public void run() {
    while (!interrupted()) {
      try {
        checkAndConfigure();
        sleep(delay);
      } catch (InterruptedException var2) {
        return;
      }
    }
  }

  private void doOnChange() {
    doOnChange.run();
  }
}
