/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.observability;

import static org.mule.runtime.api.util.MuleSystemProperties.ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY;
import static org.mule.runtime.api.util.MuleSystemProperties.ENABLE_TRACER_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY;

import static java.lang.Boolean.getBoolean;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.nio.file.Path.of;

import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfiguration;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfigurationFileFinder;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfigurationPropertyResolver;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;

/**
 * A class that provides functionality to read configuration from a file.
 *
 * @since 4.6.0
 */
public abstract class AbstractFileObservabilitySignalConfiguration implements ObservabilitySignalConfiguration {

  private static final ObjectMapper configFileMapper = new ObjectMapper(new YAMLFactory());
  private static final String UNSUPPORTED_TYPE_ERROR_MESSAGE =
      "Configuration key '%s' has an unsupported type %s. Only string values are allowed.";
  private final ObservabilitySignalConfigurationPropertyResolver observabilitySignalConfigurationPropertyResolver;
  private boolean isPropertiesInitialized = false;
  private JsonNode configuration;
  private File configurationFile;
  private final ObservabilitySignalConfigurationFileFinder artifactResourceFinder;

  protected AbstractFileObservabilitySignalConfiguration(ObservabilitySignalConfigurationFileFinder artifactResourceFinder,
                                                         ObservabilitySignalConfigurationPropertyResolver observabilitySignalConfigurationPropertyResolver) {
    this.observabilitySignalConfigurationPropertyResolver = observabilitySignalConfigurationPropertyResolver;
    this.artifactResourceFinder = artifactResourceFinder;
  }

  /**
   * Invoked when the configuration file is not found.
   */
  protected abstract void onConfigurationFileNotFound();

  /**
   * Invoked when an error during configuration file loading occurs.
   *
   * @param error             The error.
   * @param configurationFile The failed configuration file.
   */
  protected abstract void onConfigurationFileLoadError(Exception error, File configurationFile);

  /**
   * @return The name of the main signal configuration file (additional files can be referenced from this configuration, such as a
   *         TLS certificate).
   */
  protected abstract String getSignalConfigurationFileName();

  /**
   * @return Absolute path to the signal configuration files directory (does not include a file name).
   */
  protected abstract Path getSignalConfigurationFileDirectoryPath();

  protected static File findArtifactConfigFile(ClassLoader executionClassloader, String configFilePath) {
    try {
      URL resource = executionClassloader.getResource(configFilePath);
      return resource != null ? new File(resource.toURI()) : null;
    } catch (URISyntaxException e) {
      return null;
    }
  }

  @Override
  public String getStringValue(String configurationKey) {
    if (!isPropertiesInitialized) {
      initialise();
      isPropertiesInitialized = true;
    }
    String configurationValue = readStringFromConfigOrSystemProperty(configurationKey);
    if (configurationValue != null) {
      // Resolves the actual configurationValue when the current one is a property reference.
      configurationValue = observabilitySignalConfigurationPropertyResolver.resolve(configurationValue);
    }
    return configurationValue;
  }

  @Override
  public Path getPathValue(String key, Path defaultValue) {
    return getAbsolutePath(ObservabilitySignalConfiguration.super.getPathValue(key, defaultValue));
  }

  protected void initialise() {
    loadConfiguration();
  }

  private void loadConfiguration() {
    try {
      configurationFile = getSignalConfigurationFromArtifactOrFromFileSystem();
      if (configurationFile.exists()) {
        parseConfiguration(configurationFile);
      } else {
        onConfigurationFileNotFound();
      }
    } catch (Exception configurationException) {
      onConfigurationFileLoadError(configurationException, configurationFile);
    }
  }

  private File getSignalConfigurationFromArtifactOrFromFileSystem() {
    return getSignalConfigurationResourceFromArtifactOrFromFileSystem(of(getSignalConfigurationFileName()));
  }

  private File getSignalConfigurationResourceFromArtifactOrFromFileSystem(Path path) {
    File configurationFile = null;
    if (isApplicationLevelConfigurable()) {
      // This delegates into artifact level signal configuration. Artifact signal configuration must be at the root of the
      // artifact resources.
      configurationFile = artifactResourceFinder.getResource(path.toString());
    }
    // This searches based on the directory of the main signal configuration file when there is not artifact level configuration.
    return configurationFile != null ? configurationFile : getSignalConfigurationFileDirectoryPath().resolve(path).toFile();
  }

  /**
   * @return True if the configuration file can be part of a mule application resources.
   */
  protected static boolean isApplicationLevelConfigurable() {
    return getBoolean(ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY)
        || getBoolean(ENABLE_TRACER_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY);
  }

  private String readStringFromConfigOrSystemProperty(String key) {
    // If the configuration is not initialized, return the system property value.
    if (configuration == null) {
      return getProperty(key);
    }

    JsonNode node = readFromConfiguration(key, configuration);
    if (node == null || node.isNull()) {
      return null;
    }
    if (!node.isValueNode()) {
      throw new IllegalArgumentException(format(UNSUPPORTED_TYPE_ERROR_MESSAGE, key, node.getNodeType()));
    }
    String value = node.asText();
    return !value.isEmpty() ? value : null;
  }

  private JsonNode readFromConfiguration(String key, JsonNode configuration) {
    String[] path = key.split("\\.");
    JsonNode node = configuration;
    for (String pathPart : path) {
      node = node.get(pathPart);
      if (node == null) {
        return null;
      }
    }
    return node;
  }

  private Path getAbsolutePath(Path path) {
    if (path.isAbsolute()) {
      return path;
    }
    // We need to resolve the non-absolute path because it can be either at the artifact resources or at an external directory.
    File configurationFile = getSignalConfigurationResourceFromArtifactOrFromFileSystem(path);
    return configurationFile.toPath().toAbsolutePath();
  }

  private void parseConfiguration(File configurationFile) throws IOException {
    configuration = configFileMapper.readTree(configurationFile);
  }

  public File getConfigurationFile() {
    return configurationFile;
  }
}
