/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.config.internal.context;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.CONNECTION;
import static org.mule.runtime.api.util.MuleSystemProperties.ENABLE_EXTRACT_CONNECTION_DATA;
import static org.mule.runtime.api.util.MuleSystemProperties.SILENT_ERRORS_EXTRACT_CONNECTION_DATA;

import static java.lang.Boolean.getBoolean;
import static java.lang.System.lineSeparator;
import static java.net.URI.create;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.lang3.Strings.CI;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.container.api.MuleFoldersUtil;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Obtains connections data from the application and generates a report with it.
 *
 * @since 4.11, 4.10, 4.9.10, 4.6.23
 */
public class ConnectionsDataExtractor {

  private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionsDataExtractor.class);

  private final String artifactName;
  private final List<ConnectionData> connections;

  public ConnectionsDataExtractor(ArtifactAst artifactAst) {
    artifactName = artifactAst.getArtifactName();

    if (!getBoolean(ENABLE_EXTRACT_CONNECTION_DATA)) {
      connections = new ArrayList<>();
      return;
    }

    Set<ComponentAst> seenConnectionLocations = new HashSet<>();

    connections = artifactAst.recursiveStream()
        .peek(comp -> {
          if (CONNECTION.equals(comp.getComponentType())) {
            seenConnectionLocations.add(comp);
          }
        })
        .filter(comp -> comp.getModel(ParameterizedModel.class).isPresent())
        .map(comp -> {
          String host = null;
          String port = "-1";

          for (ComponentParameterAst param : comp.getParameters()) {
            if (param.getValue() == null || param.getValue().getRight() == null) {
              continue;
            }
            try {
              final var paramName = param.getModel().getName();
              final var paramValue = param.getValue().getRight();

              if (CI.contains(paramName, "url") || CI.contains(paramName, "uri")) {
                try {
                  final var uriParam = create(paramValue.toString());
                  host = uriParam.getHost();
                  final var portFromUri = uriParam.getPort();
                  if (portFromUri == -1) {
                    final var scheme = uriParam.getScheme();
                    if ("http".equals(scheme)) {
                      port = "80";
                    } else if ("https".equals(scheme)) {
                      port = "443";
                    }
                  } else {
                    port = "" + portFromUri;
                  }
                } catch (IllegalArgumentException e) {
                  // ignoring invalid url param
                  LOGGER.warn("Exception {} parsing url {}", e, paramValue);
                }
              }
              if (CI.contains(paramName, "host")) {
                host = paramValue.toString();
              }
              if (CI.contains(paramName, "port")) {
                port = paramValue.toString();
              }

              if ("0.0.0.0".equals(host)) {
                host = "localhost";
              }
            } catch (Exception e) {
              LOGGER.error("Exception thrown for param " + param.getModel().getName() + " @ " + comp.getLocation().getLocation(),
                           e);
              if (getBoolean(SILENT_ERRORS_EXTRACT_CONNECTION_DATA)) {
                return Optional.<ConnectionData>empty();
              } else {
                throw e;
              }
            }
          }

          if (host != null) {
            return of(new ConnectionData(comp.getLocation().getLocation(),
                                         comp.getIdentifier().toString(),
                                         host, port));
          } else {
            return Optional.<ConnectionData>empty();
          }
        })
        .filter(Optional::isPresent)
        .map(Optional::get)
        .collect(toList());

    // for connections providers for which data wasn't found, add them to the report so they can be rechecked manually.
    seenConnectionLocations
        .stream()
        .filter(seenConn -> connections.stream()
            .noneMatch(conn -> conn.location.startsWith(seenConn.getLocation().getLocation())))
        .map(seenConn -> new ConnectionData(seenConn.getLocation().getLocation(),
                                            seenConn.getIdentifier().toString(),
                                            "", "-1"))
        .forEach(connections::add);
  }

  public void persist() {
    if (!getBoolean(ENABLE_EXTRACT_CONNECTION_DATA)) {
      return;
    }

    final var muleBaseFolder = MuleFoldersUtil.getMuleBaseFolder();

    final var tlsDir = new File(muleBaseFolder, ".mule/.introspection");
    tlsDir.mkdirs();
    final var fileName = "connections_data_%s.csv".formatted(artifactName.replaceAll("[:\\\\/*\"?|<>']", "_"));
    final var targetFile = new File(tlsDir, fileName);

    try (BufferedWriter bufferedWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(targetFile)))) {
      bufferedWriter.write("Component,Host,Port" + lineSeparator());
      LOGGER.debug("Component,Host,Port");

      for (ConnectionData connectionData : connections) {
        bufferedWriter.write(connectionData + lineSeparator());
        LOGGER.debug("{}", connectionData);
      }
    } catch (Exception e) {
      LOGGER.error("Exception thrown when writing '" + fileName + "'", e);
      if (getBoolean(SILENT_ERRORS_EXTRACT_CONNECTION_DATA)) {
        throw new MuleRuntimeException(e);
      }
    }
  }

  private static final record ConnectionData(String location, String identifier, String host, String port) {

    @Override
    public final String toString() {
      return "\"%s(%s)\",\"%s\",\"%s\"".formatted(location, identifier, host, port);
    }
  }
}
