/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.config.internal.dsl.processor;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.mule.runtime.api.notification.NotificationDispatcher;
import org.mule.runtime.core.api.retry.RetryNotifier;
import org.mule.runtime.core.api.retry.async.AsynchronousRetryTemplate;
import org.mule.runtime.core.api.retry.policy.NoRetryPolicyTemplate;
import org.mule.runtime.core.api.retry.policy.SimpleRetryPolicyTemplate;
import org.mule.tck.util.MockInjector;

import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.Mockito.verifyNoMoreInteractions;

@ExtendWith(MockitoExtension.class)
class RetryPolicyTemplateObjectFactoryTestCase {

  private RetryPolicyTemplateObjectFactory factory;
  @Mock
  private RetryNotifier retryNotifier;
  /**
   * Injected into factory.
   */
  @Mock
  private NotificationDispatcher notificationFirer;

  @BeforeEach
  void setUp() {
    factory = new RetryPolicyTemplateObjectFactory();
    MockInjector.injectMocksFromSuite(this, factory);
  }

  @Test
  void setBlocking() throws Exception {
    factory.setBlocking(false);
    // "Blocking" means not an async retry - it defaults to true, so set to false to check....
    assertThat(factory.getObject(), instanceOf(AsynchronousRetryTemplate.class));
  }

  @Test
  void setCount() throws Exception {
    factory.setCount(0);

    assertThat(factory.getObject(), instanceOf(NoRetryPolicyTemplate.class));
  }

  @Test
  void setFrequency() throws Exception {
    factory.setFrequency(100);

    SimpleRetryPolicyTemplate result = (SimpleRetryPolicyTemplate) factory.getObject();

    assertThat(result.getFrequency(), is(100L));
  }

  @Test
  void setRetryNotifier() throws Exception {
    factory.setRetryNotifier(retryNotifier);

    SimpleRetryPolicyTemplate result = (SimpleRetryPolicyTemplate) factory.getObject();

    assertThat(result.getNotifier(), is(retryNotifier));
  }

  @Test
  void notificationFirerNotCalled() throws Exception {
    factory.getObject().createRetryInstance().applyPolicy(new IllegalArgumentException("The Crimson Capybara!"));

    verifyNoMoreInteractions(notificationFirer);
  }
}
