/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.config.internal.resolvers;

import static org.mule.runtime.ast.graph.api.ArtifactAstDependencyGraphFactory.generateFor;

import static java.util.Collections.emptySet;
import static java.util.Objects.requireNonNullElse;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.config.internal.model.ApplicationModel;

import java.util.Collection;
import java.util.Optional;

/**
 * Provides dependencies based on the information from the ArtifactAstDependencyGraph
 */
public class ConfigurationDependencyResolver {

  private final MultiMap<String, String> directComponentDependencies;

  /**
   * Creates a new instance associated to a complete {@link ApplicationModel}.
   *
   * @param applicationModel the artifact {@link ApplicationModel}.
   */
  public ConfigurationDependencyResolver(ArtifactAst applicationModel) {
    var appModelDependencyGraph = generateFor(applicationModel);

    MultiMap<String, String> dependencies = new MultiMap<>();
    applicationModel.recursiveStream()
        .map(ComponentAst::getComponentId)
        .filter(Optional::isPresent)
        .map(Optional::get)
        .forEach(componentName -> dependencies.put(componentName,
                                                   appModelDependencyGraph.getRequiredComponents(componentName)
                                                       .stream()
                                                       .map(ComponentAst::getComponentId)
                                                       .filter(Optional::isPresent)
                                                       .map(Optional::get)
                                                       .toList()));

    directComponentDependencies = dependencies.toImmutableMultiMap();
  }

  /**
   * @param componentName the name attribute value of the component
   * @return the direct dependencies of the component with component name {@code #componentName}. An empty collection if there is
   *         no component with such name.
   */
  public Collection<String> getDirectComponentDependencies(String componentName) {
    return requireNonNullElse(directComponentDependencies.getAll(componentName),
                              emptySet());
  }

}
