/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.config.internal.context.service;

import static org.mule.runtime.config.internal.context.service.InjectParamsFromContextServiceUtils.MANY_CANDIDATES_ERROR_MSG_TEMPLATE;
import static org.mule.runtime.config.internal.context.service.InjectParamsFromContextServiceUtils.NO_OBJECT_FOUND_FOR_PARAM;
import static org.mule.runtime.config.utils.Utils.augmentedParam;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_MULE_CONTEXT;

import static java.lang.String.format;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.Matchers.sameInstance;
import static org.junit.Assert.assertThrows;

import static org.mockito.Mockito.when;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.config.utils.Utils.AmbiguousAugmentedMethodService;
import org.mule.runtime.config.utils.Utils.AugmentedMethodService;
import org.mule.runtime.config.utils.Utils.AugmentedSubclassMethodService;
import org.mule.runtime.config.utils.Utils.AugmentedSubclassOverridesMethodService;
import org.mule.runtime.config.utils.Utils.BaseOverloadedService;
import org.mule.runtime.config.utils.Utils.BaseService;
import org.mule.runtime.config.utils.Utils.BasicService;
import org.mule.runtime.config.utils.Utils.HiddenAugmentedMethodService;
import org.mule.runtime.config.utils.Utils.InvalidAugmentedMethodService;
import org.mule.runtime.config.utils.Utils.InvalidNamedAugmentedMethodService;
import org.mule.runtime.config.utils.Utils.NamedAugmentedMethodService;
import org.mule.runtime.config.utils.Utils.OverloadedAugmentedMethodService;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.registry.IllegalDependencyInjectionException;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.Optional;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

import jakarta.inject.Inject;

@ExtendWith(MockitoExtension.class)
class InjectParamsFromContextServiceMethodInvokerTestCase extends AbstractMuleTestCase {

  @Mock
  private Registry registry;

  @Mock
  private MuleContext muleContext;

  private InjectParamsFromContextServiceMethodInvoker injectParamsFromContextServiceMethodInvoker;
  private Method method;

  @BeforeEach
  public void setUp() throws NoSuchMethodException {
    injectParamsFromContextServiceMethodInvoker = new InjectParamsFromContextServiceMethodInvoker(registry);
    method = BaseService.class.getMethod("augmented");
    augmentedParam = null;
  }

  @Test
  void notAugmentedInvocation() throws Throwable {
    BaseService service = new BasicService();

    injectParamsFromContextServiceMethodInvoker.invoke(service, method, null);

    assertThat(augmentedParam, is(true));
  }

  @Test
  void augmentedInvocation() throws Throwable {
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseService service = new AugmentedMethodService();

    injectParamsFromContextServiceMethodInvoker.invoke(service, method, null);

    assertThat(augmentedParam, sameInstance(muleContext));
  }

  @Test
  void augmentedSubclassInvocation() throws Throwable {
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseService service = new AugmentedSubclassMethodService();

    injectParamsFromContextServiceMethodInvoker.invoke(service, method, null);

    assertThat(augmentedParam, sameInstance(muleContext));
  }

  @Test
  void augmentedSubclassOverridesInvocation() throws Throwable {
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseService service = new AugmentedSubclassOverridesMethodService();

    injectParamsFromContextServiceMethodInvoker.invoke(service, method, null);

    assertThat(augmentedParam, is(true));
  }

  @Test
  void namedAugmentedInvocation() throws Throwable {
    when(registry.lookupByName(OBJECT_MULE_CONTEXT)).thenReturn(of(muleContext));
    BaseService service = new NamedAugmentedMethodService();

    injectParamsFromContextServiceMethodInvoker.invoke(service, method, null);

    assertThat(augmentedParam, sameInstance(muleContext));
  }

  @Test
  void invalidNamedAugmentedInvocation() throws Throwable {
    BaseService service = new InvalidNamedAugmentedMethodService();

    var thrown = assertThrows(IllegalDependencyInjectionException.class,
                              () -> injectParamsFromContextServiceMethodInvoker.invoke(service, method, null));
    assertThat(thrown.getMessage(), is(format(NO_OBJECT_FOUND_FOR_PARAM, "param", method.getName(), service.toString())));
  }

  @Test
  void hiddenAugmentedInvocation() throws Throwable {
    BaseService service = new HiddenAugmentedMethodService();

    injectParamsFromContextServiceMethodInvoker.invoke(service, method, null);

    assertThat(augmentedParam, is(true));
  }

  @Test
  void overloadedAugmentedInvocation() throws Throwable {
    when(registry.lookupAllByType(int.class)).thenReturn(emptyList());
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseOverloadedService service = new OverloadedAugmentedMethodService();

    injectParamsFromContextServiceMethodInvoker.invoke(service, method, null);

    assertThat(augmentedParam, is(true));
  }

  @Test
  void overloadedAugmentedInvocation2() throws Throwable {
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseOverloadedService service = new OverloadedAugmentedMethodService();

    java.util.List<Method> methods = asList(OverloadedAugmentedMethodService.class.getMethods());

    Optional<Method> method = methods.stream().filter(m -> m.getName().equals("augmented") && m.getParameterCount() == 1
        && !m.getParameters()[0].getName().contains("context")).findFirst();

    injectParamsFromContextServiceMethodInvoker.invoke(service, method.get(), new Object[] {1});

    assertThat(augmentedParam, sameInstance(muleContext));
  }

  @Test
  void ambiguousAugmentedInvocation() throws Throwable {
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseService service = new AmbiguousAugmentedMethodService();

    var thrown = assertThrows(IllegalDependencyInjectionException.class,
                              () -> injectParamsFromContextServiceMethodInvoker.invoke(service, method, null));
    assertThat(thrown.getMessage(), is(format(MANY_CANDIDATES_ERROR_MSG_TEMPLATE, method.getName(), service.toString())));

    assertThat(augmentedParam, nullValue());
  }

  @Test
  void invalidAugmentedInvocation() throws Throwable {
    BaseService service = new InvalidAugmentedMethodService();

    injectParamsFromContextServiceMethodInvoker.invoke(service, method, null);

    assertThat(augmentedParam, is(true));
  }

}
