/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.config.internal.context;

import static java.util.Map.entry;
import static java.util.Map.of;
import static java.util.Map.ofEntries;
import static org.mule.runtime.api.cluster.ClusterService.OBJECT_CLUSTER_SERVICE;
import static org.mule.runtime.api.lock.LockProvider.OBJECT_LOCK_PROVIDER;
import static org.mule.runtime.api.store.ObjectStoreManager.BASE_IN_MEMORY_OBJECT_STORE_KEY;
import static org.mule.runtime.api.store.ObjectStoreManager.BASE_PERSISTENT_OBJECT_STORE_KEY;
import static org.mule.runtime.config.api.LazyComponentInitializer.LAZY_COMPONENT_INITIALIZER_SERVICE_KEY;
import static org.mule.runtime.core.api.config.MuleDeploymentProperties.DEPLOYMENT_PROPERTY_PREFIX;
import static org.mule.runtime.core.api.config.MuleDeploymentProperties.MULE_ADD_ARTIFACT_AST_TO_REGISTRY_DEPLOYMENT_PROPERTY;
import static org.mule.runtime.core.api.config.MuleProperties.FORWARD_COMPATIBILITY_HELPER_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.INTERCEPTOR_MANAGER_REGISTRY_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.LOCAL_OBJECT_LOCK_FACTORY;
import static org.mule.runtime.core.api.config.MuleProperties.LOCAL_OBJECT_STORE_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HTTP_SERVICE_API_REGISTRY_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_ARTIFACT_AST;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_COMPONENT_INITIAL_STATE_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_CONFIGURATION_PROPERTIES;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_CONNECTION_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_CONNECTIVITY_TESTER_FACTORY;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_CONVERTER_RESOLVER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_DEFAULT_MESSAGE_PROCESSING_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_EXCEPTION_LOCATION_PROVIDER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_EXTENSION_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_FLOW_CLASSIFIER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_LOCAL_QUEUE_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_LOCAL_STORE_IN_MEMORY;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_LOCAL_STORE_PERSISTENT;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_LOCK_FACTORY;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_MESSAGE_PROCESSING_FLOW_TRACE_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_MULE_CONFIGURATION;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_MULE_CONTEXT;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_MULE_STREAM_CLOSER_SERVICE;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_NOTIFICATION_DISPATCHER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_NOTIFICATION_HANDLER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_POLICY_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_PROCESSING_TIME_WATCHER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_QUEUE_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_RESOURCE_LOCATOR;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_SECURITY_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_STATISTICS;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_STORE_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_TRANSACTION_FACTORY_LOCATOR;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_TRANSACTION_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_TRANSFORMATION_SERVICE;
import static org.mule.runtime.core.api.config.MuleProperties.SDK_OBJECT_STORE_MANAGER;
import static org.mule.runtime.core.api.config.bootstrap.ArtifactType.DOMAIN;
import static org.mule.runtime.core.internal.config.bootstrap.AbstractRegistryBootstrap.BINDING_PROVIDER_PREDICATE;
import static org.mule.runtime.core.internal.config.bootstrap.AbstractRegistryBootstrap.TRANSFORMER_PREDICATE;
import static org.mule.runtime.core.internal.el.function.MuleFunctionsBindingContextProvider.CORE_FUNCTIONS_PROVIDER_REGISTRY_KEY;
import static org.mule.runtime.feature.api.management.FeatureFlaggingManagementService.PROFILING_FEATURE_MANAGEMENT_SERVICE_KEY;

import static java.lang.Boolean.parseBoolean;

import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.notification.NotificationListenerRegistry;
import org.mule.runtime.api.queue.QueueManager;
import org.mule.runtime.api.util.ResourceLocator;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.config.internal.factories.ConstantFactoryBean;
import org.mule.runtime.config.internal.factories.ExtensionManagerFactoryBean;
import org.mule.runtime.config.internal.factories.MuleConfigurationConfigurator;
import org.mule.runtime.config.internal.factories.MuleContextFactoryBean;
import org.mule.runtime.config.internal.factories.TransactionManagerFactoryBean;
import org.mule.runtime.config.internal.model.dsl.config.DefaultComponentInitialStateManager;
import org.mule.runtime.config.internal.registry.SpringRegistryBootstrap;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.runtime.core.api.event.EventContextService;
import org.mule.runtime.core.internal.cluster.DefaultClusterService;
import org.mule.runtime.core.internal.config.CustomService;
import org.mule.runtime.core.internal.config.DefaultFeatureManagementService;
import org.mule.runtime.core.internal.config.InternalCustomizationService;
import org.mule.runtime.core.internal.connection.DefaultConnectivityTesterFactory;
import org.mule.runtime.core.internal.connection.DelegateConnectionManagerAdapter;
import org.mule.runtime.core.internal.context.MuleContextWithRegistry;
import org.mule.runtime.core.internal.context.notification.DefaultNotificationDispatcher;
import org.mule.runtime.core.internal.context.notification.DefaultNotificationListenerRegistry;
import org.mule.runtime.core.internal.context.notification.MessageProcessingFlowTraceManager;
import org.mule.runtime.core.internal.el.function.MuleFunctionsBindingContextProvider;
import org.mule.runtime.core.internal.event.DefaultEventContextService;
import org.mule.runtime.core.internal.exception.MessagingExceptionLocationProvider;
import org.mule.runtime.core.internal.execution.MuleMessageProcessingManager;
import org.mule.runtime.core.internal.lock.MuleLockFactory;
import org.mule.runtime.core.internal.lock.SingleServerLockProvider;
import org.mule.runtime.core.internal.management.stats.DefaultProcessingTimeWatcher;
import org.mule.runtime.core.internal.policy.DefaultPolicyManager;
import org.mule.runtime.core.internal.processor.interceptor.DefaultProcessorInterceptorManager;
import org.mule.runtime.core.internal.security.DefaultMuleSecurityManager;
import org.mule.runtime.core.internal.transaction.TransactionFactoryLocator;
import org.mule.runtime.core.internal.transformer.DynamicDataTypeConversionResolver;
import org.mule.runtime.core.internal.transformer.ExtendedTransformationService;
import org.mule.runtime.core.internal.util.DefaultStreamCloserService;
import org.mule.runtime.core.internal.util.queue.TransactionalQueueManager;
import org.mule.runtime.core.internal.util.queue.wrapper.LocalQueueManagerWrapper;
import org.mule.runtime.core.internal.util.queue.wrapper.QueueManagerWrapper;
import org.mule.runtime.core.internal.util.store.DefaultObjectStoreFactoryBean;
import org.mule.runtime.core.internal.util.store.MuleObjectStoreManager;
import org.mule.runtime.http.support.api.HttpServiceApiDelegate;
import org.mule.runtime.module.extension.api.runtime.compatibility.DefaultForwardCompatibilityHelper;
import org.mule.runtime.module.extension.internal.store.SdkObjectStoreManagerAdapter;

import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;

import jakarta.inject.Inject;

/**
 * This class configured all the services available in a {@code MuleContext}.
 * <p>
 * This class takes cares of registering bean definitions for each of the provided services so dependency injection can be
 * properly done through the use of {@link Inject}.
 *
 * @since 4.0
 */
public class SpringMuleContextServiceConfigurator extends AbstractSpringMuleContextServiceConfigurator {

  private static final Logger LOGGER = LoggerFactory.getLogger(SpringMuleContextServiceConfigurator.class);

  private static final Map<String, String> OBJECT_STORE_NAME_TO_LOCAL_OBJECT_STORE_NAME =
      of(BASE_IN_MEMORY_OBJECT_STORE_KEY, OBJECT_LOCAL_STORE_IN_MEMORY,
         BASE_PERSISTENT_OBJECT_STORE_KEY, OBJECT_LOCAL_STORE_PERSISTENT);

  private final MuleContextWithRegistry muleContext;
  private final ArtifactType artifactType;
  private final ArtifactAst artifactAst;
  private final ResourceLocator resourceLocator;

  // Do not use static field. BeanDefinitions are reused and produce weird behaviour
  private final Map<String, BeanDefinition> defaultContextServices = ofEntries(
                                                                               entry(OBJECT_TRANSACTION_MANAGER,
                                                                                     getBeanDefinition(TransactionManagerFactoryBean.class)),
                                                                               entry(OBJECT_EXTENSION_MANAGER,
                                                                                     getBeanDefinition(ExtensionManagerFactoryBean.class)),
                                                                               entry(OBJECT_CONNECTION_MANAGER,
                                                                                     getBeanDefinition(DelegateConnectionManagerAdapter.class)),
                                                                               entry(OBJECT_MULE_CONFIGURATION,
                                                                                     getBeanDefinition(MuleConfigurationConfigurator.class)),
                                                                               entry(OBJECT_TRANSACTION_FACTORY_LOCATOR,
                                                                                     getBeanDefinition(TransactionFactoryLocator.class)),
                                                                               entry(OBJECT_POLICY_MANAGER,
                                                                                     getBeanDefinition(DefaultPolicyManager.class)),
                                                                               entry(INTERCEPTOR_MANAGER_REGISTRY_KEY,
                                                                                     getBeanDefinition(DefaultProcessorInterceptorManager.class)),
                                                                               entry(OBJECT_NOTIFICATION_DISPATCHER,
                                                                                     getBeanDefinition(DefaultNotificationDispatcher.class)),
                                                                               entry(NotificationListenerRegistry.REGISTRY_KEY,
                                                                                     getBeanDefinition(DefaultNotificationListenerRegistry.class)),
                                                                               entry(EventContextService.REGISTRY_KEY,
                                                                                     getBeanDefinition(DefaultEventContextService.class)),
                                                                               entry(BASE_IN_MEMORY_OBJECT_STORE_KEY,
                                                                                     getBeanDefinitionBuilder(ConstantFactoryBean.class)
                                                                                         .addConstructorArgReference(OBJECT_LOCAL_STORE_IN_MEMORY)
                                                                                         .getBeanDefinition()),
                                                                               entry(OBJECT_LOCAL_STORE_IN_MEMORY,
                                                                                     getBeanDefinition(DefaultObjectStoreFactoryBean.class,
                                                                                                       "createDefaultInMemoryObjectStore")),
                                                                               entry(BASE_PERSISTENT_OBJECT_STORE_KEY,
                                                                                     getBeanDefinitionBuilder(ConstantFactoryBean.class)
                                                                                         .addConstructorArgReference(OBJECT_LOCAL_STORE_PERSISTENT)
                                                                                         .getBeanDefinition()),
                                                                               entry(OBJECT_LOCAL_STORE_PERSISTENT,
                                                                                     getBeanDefinition(DefaultObjectStoreFactoryBean.class,
                                                                                                       "createDefaultPersistentObjectStore")),
                                                                               entry(OBJECT_STORE_MANAGER,
                                                                                     getPrimaryBeanDefinition(MuleObjectStoreManager.class)),
                                                                               entry(SDK_OBJECT_STORE_MANAGER,
                                                                                     getPrimaryBeanDefinition(SdkObjectStoreManagerAdapter.class)),
                                                                               entry(QueueManager.OBJECT_QUEUE_MANAGER,
                                                                                     getBeanDefinition(TransactionalQueueManager.class)),
                                                                               entry(OBJECT_QUEUE_MANAGER,
                                                                                     getBeanDefinition(QueueManagerWrapper.class)),
                                                                               entry(OBJECT_SECURITY_MANAGER,
                                                                                     getBeanDefinition(DefaultMuleSecurityManager.class)),
                                                                               entry(OBJECT_DEFAULT_MESSAGE_PROCESSING_MANAGER,
                                                                                     getBeanDefinition(MuleMessageProcessingManager.class)),
                                                                               entry(OBJECT_MULE_STREAM_CLOSER_SERVICE,
                                                                                     getBeanDefinition(DefaultStreamCloserService.class)),
                                                                               entry(OBJECT_CONVERTER_RESOLVER,
                                                                                     getBeanDefinition(DynamicDataTypeConversionResolver.class)),
                                                                               entry(OBJECT_LOCK_FACTORY,
                                                                                     getBeanDefinition(MuleLockFactory.class,
                                                                                                       true)),
                                                                               entry(OBJECT_LOCK_PROVIDER,
                                                                                     getBeanDefinition(SingleServerLockProvider.class)),
                                                                               entry(OBJECT_PROCESSING_TIME_WATCHER,
                                                                                     getBeanDefinition(DefaultProcessingTimeWatcher.class)),
                                                                               entry(OBJECT_EXCEPTION_LOCATION_PROVIDER,
                                                                                     getBeanDefinition(MessagingExceptionLocationProvider.class)),
                                                                               entry(OBJECT_MESSAGE_PROCESSING_FLOW_TRACE_MANAGER,
                                                                                     getBeanDefinition(MessageProcessingFlowTraceManager.class)),
                                                                               entry(OBJECT_COMPONENT_INITIAL_STATE_MANAGER,
                                                                                     getBeanDefinition(DefaultComponentInitialStateManager.class)),
                                                                               entry(OBJECT_TRANSFORMATION_SERVICE,
                                                                                     getBeanDefinition(ExtendedTransformationService.class)),
                                                                               entry(OBJECT_CLUSTER_SERVICE,
                                                                                     getBeanDefinition(DefaultClusterService.class)),
                                                                               entry(OBJECT_CONNECTIVITY_TESTER_FACTORY,
                                                                                     getBeanDefinition(DefaultConnectivityTesterFactory.class)),
                                                                               entry(LAZY_COMPONENT_INITIALIZER_SERVICE_KEY,
                                                                                     getBeanDefinition(NoOpLazyComponentInitializer.class)),
                                                                               entry(PROFILING_FEATURE_MANAGEMENT_SERVICE_KEY,
                                                                                     getBeanDefinition(DefaultFeatureManagementService.class)),
                                                                               entry(FORWARD_COMPATIBILITY_HELPER_KEY,
                                                                                     getBeanDefinition(DefaultForwardCompatibilityHelper.class)),
                                                                               entry(MULE_HTTP_SERVICE_API_REGISTRY_KEY,
                                                                                     getBeanDefinition(HttpServiceApiDelegate.class)));

  private final MuleFunctionsBindingContextProvider coreFunctionsProvider;
  private final ConfigurationProperties configurationProperties;
  private final Map<String, String> artifactProperties;

  public SpringMuleContextServiceConfigurator(MuleContextWithRegistry muleContext,
                                              MuleFunctionsBindingContextProvider coreFunctionsProvider,
                                              ConfigurationProperties configurationProperties,
                                              Map<String, String> artifactProperties,
                                              ArtifactType artifactType,
                                              ArtifactAst artifactAst,
                                              BeanDefinitionRegistry beanDefinitionRegistry,
                                              ResourceLocator resourceLocator) {
    super((InternalCustomizationService) muleContext.getCustomizationService(), beanDefinitionRegistry);

    this.muleContext = muleContext;
    this.coreFunctionsProvider = coreFunctionsProvider;
    this.configurationProperties = configurationProperties;
    this.artifactProperties = artifactProperties;
    this.artifactType = artifactType;
    this.artifactAst = artifactAst;
    this.resourceLocator = resourceLocator;
  }

  protected void createArtifactServices() {
    new ConnectionsDataExtractor(artifactAst).persist();

    registerBeanDefinition(OBJECT_MULE_CONTEXT, createMuleContextDefinition());
    // This configurationProperties will contain the properties configured in the app itself, taking precedence over the base one
    // from the base registry
    registerConstantBeanDefinition(OBJECT_CONFIGURATION_PROPERTIES, configurationProperties);
    registerConstantBeanDefinition(OBJECT_NOTIFICATION_HANDLER, muleContext.getNotificationManager());
    registerConstantBeanDefinition(OBJECT_FLOW_CLASSIFIER, new FlowClassifierFactory(artifactAst).create());
    registerConstantBeanDefinition(OBJECT_STATISTICS, muleContext.getStatistics());
    registerConstantBeanDefinition(OBJECT_RESOURCE_LOCATOR, resourceLocator);
    if (!artifactType.equals(DOMAIN)) {
      loadServiceConfigurators();
    }

    registerContextServices(defaultContextServices, artifactType.equals(DOMAIN));

    createBootstrapBeanDefinitions();
    createLocalObjectStoreBeanDefinitions();
    createLocalLockFactoryBeanDefinitions();
    createQueueManagerBeanDefinitions();
    createCustomServices();

    coreFunctionsProvider.setConfigurationProperties(configurationProperties);
    registerConstantBeanDefinition(CORE_FUNCTIONS_PROVIDER_REGISTRY_KEY, coreFunctionsProvider);

    getArtifactProperties()
        .entrySet()
        .stream()
        .filter(e -> !e.getKey().startsWith(DEPLOYMENT_PROPERTY_PREFIX))
        .forEach(e -> registerConstantBeanDefinition(e.getKey(), e.getValue()));

    if (parseBoolean(getArtifactProperties().get(MULE_ADD_ARTIFACT_AST_TO_REGISTRY_DEPLOYMENT_PROPERTY))) {
      registerConstantBeanDefinition(OBJECT_ARTIFACT_AST, artifactAst);
    }
  }

  private void createCustomServices() {
    final Map<String, CustomService> customServices = getCustomizationService().getCustomServices();
    for (String serviceName : customServices.keySet()) {

      if (containsBeanDefinition(serviceName)) {
        throw new IllegalStateException("There is already a bean definition registered with key: " + serviceName);
      }

      final CustomService customService = customServices.get(serviceName);

      if (!customService.isBaseContext()
          // TODO MULE-19927 avoid this filter
          && !isServiceRuntimeProvided(customService)) {
        final BeanDefinition beanDefinition = getCustomServiceBeanDefinition(customService, serviceName);

        LOGGER.debug("Registering runtime service '{}' for {}...", serviceName, artifactType.name());
        registerBeanDefinition(serviceName, beanDefinition);
      }
    }
  }

  private void createQueueManagerBeanDefinitions() {
    AtomicBoolean customManagerDefined = new AtomicBoolean(false);
    getCustomizationService().getOverriddenService(OBJECT_QUEUE_MANAGER).ifPresent(customService -> {
      customManagerDefined.set(true);
      registerBeanDefinition(OBJECT_QUEUE_MANAGER, getCustomServiceBeanDefinition(customService, OBJECT_QUEUE_MANAGER));
    });
    getCustomizationService().getOverriddenService(QueueManager.OBJECT_QUEUE_MANAGER).ifPresent(customService -> {
      customManagerDefined.set(true);
      registerBeanDefinition(QueueManager.OBJECT_QUEUE_MANAGER,
                             getCustomServiceBeanDefinition(customService, QueueManager.OBJECT_QUEUE_MANAGER));
    });

    if (customManagerDefined.get()) {
      registerBeanDefinition(OBJECT_LOCAL_QUEUE_MANAGER, getBeanDefinition(LocalQueueManagerWrapper.class));
    } else {
      getBeanDefinitionRegistry().registerAlias(OBJECT_QUEUE_MANAGER, OBJECT_LOCAL_QUEUE_MANAGER);
    }
  }

  private void createLocalLockFactoryBeanDefinitions() {
    AtomicBoolean customLockFactoryWasDefined = new AtomicBoolean(false);
    getCustomizationService().getOverriddenService(OBJECT_LOCK_FACTORY).ifPresent(customService -> {
      customLockFactoryWasDefined.set(true);
      getBeanDefinitionRegistry().registerBeanDefinition(OBJECT_LOCK_FACTORY,
                                                         getCustomServiceBeanDefinition(customService, OBJECT_LOCK_FACTORY));
    });

    if (customLockFactoryWasDefined.get()) {
      getBeanDefinitionRegistry()
          .registerBeanDefinition(LOCAL_OBJECT_LOCK_FACTORY, defaultContextServices.get(OBJECT_LOCK_FACTORY));
    } else {
      getBeanDefinitionRegistry().registerAlias(OBJECT_LOCK_FACTORY, LOCAL_OBJECT_LOCK_FACTORY);
    }
  }

  private void createLocalObjectStoreBeanDefinitions() {
    AtomicBoolean anyBaseStoreWasRedefined = new AtomicBoolean(false);
    OBJECT_STORE_NAME_TO_LOCAL_OBJECT_STORE_NAME.entrySet().forEach(objectStoreLocal -> getCustomizationService()
        .getOverriddenService(objectStoreLocal.getKey()).ifPresent(customService -> {
          anyBaseStoreWasRedefined.set(true);
          getBeanDefinitionRegistry().registerBeanDefinition(objectStoreLocal.getKey(),
                                                             getCustomServiceBeanDefinition(customService,
                                                                                            objectStoreLocal
                                                                                                .getKey()));
        }));

    if (anyBaseStoreWasRedefined.get()) {
      final AbstractBeanDefinition beanDefinition = getBeanDefinitionBuilder(MuleObjectStoreManager.class)
          .addPropertyValue("basePersistentStoreKey", OBJECT_LOCAL_STORE_PERSISTENT)
          .addPropertyValue("baseTransientStoreKey", OBJECT_LOCAL_STORE_IN_MEMORY)
          .getBeanDefinition();
      beanDefinition.setPrimary(false);
      getBeanDefinitionRegistry().registerBeanDefinition(LOCAL_OBJECT_STORE_MANAGER, beanDefinition);
    } else {
      getBeanDefinitionRegistry().registerAlias(OBJECT_STORE_MANAGER, LOCAL_OBJECT_STORE_MANAGER);
    }
  }

  private BeanDefinition createMuleContextDefinition() {
    return getBeanDefinitionBuilder(MuleContextFactoryBean.class)
        .addConstructorArgValue(muleContext)
        .getBeanDefinition();
  }

  protected void createBootstrapBeanDefinitions() {
    try {
      SpringRegistryBootstrap springRegistryBootstrap =
          new SpringRegistryBootstrap(artifactType.getArtifactType(),
                                      muleContext.getRegistryBootstrapServiceDiscoverer(),
                                      this::registerBeanDefinition,
                                      BINDING_PROVIDER_PREDICATE
                                          .or(TRANSFORMER_PREDICATE)
                                          .negate());
      springRegistryBootstrap.initialise();
    } catch (InitialisationException e) {
      throw new RuntimeException(e);
    }
  }

  private static BeanDefinition getPrimaryBeanDefinition(Class<?> beanType) {
    BeanDefinition beanDefinition = getBeanDefinition(beanType);
    beanDefinition.setPrimary(true);

    return beanDefinition;
  }

  private static BeanDefinition getBeanDefinition(Class<?> beanType, String factoryMethodName) {
    return getBeanDefinitionBuilder(beanType).setFactoryMethod(factoryMethodName).getBeanDefinition();
  }

  protected MuleContextWithRegistry getMuleContext() {
    return muleContext;
  }

  protected Map<String, String> getArtifactProperties() {
    return artifactProperties;
  }

}
