/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.config.api.dsl.processor.xml;

import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.config.internal.dsl.xml.StaticXmlNamespaceInfo;
import org.mule.runtime.config.internal.dsl.xml.StaticXmlNamespaceInfoProvider;
import org.mule.runtime.core.api.registry.AbstractServiceRegistry;
import org.mule.runtime.core.api.registry.ServiceRegistry;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfo;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfoProvider;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static java.util.List.copyOf;
import static java.util.List.of;
import static org.mule.runtime.api.util.collection.Collectors.toImmutableList;

/**
 * Custom service that extends the list of {@link XmlNamespaceInfoProvider}s discovered using a {@link ServiceRegistry} with a
 * {@link XmlNamespaceInfoProvider} providing information of the namespaces for the {@link ExtensionModel extensions} in the
 * current context.
 *
 * @since 4.0
 */
public final class XmlApplicationServiceRegistry extends AbstractServiceRegistry {

  private final ServiceRegistry delegate;
  private final XmlNamespaceInfoProvider extensionsXmlInfoProvider;

  public XmlApplicationServiceRegistry(ServiceRegistry delegate, DslResolvingContext context) {
    this.delegate = delegate;

    List<XmlNamespaceInfo> extensionNamespaces;
    if (context != null) {
      extensionNamespaces = context.getExtensions().stream()
          .map(ext -> new StaticXmlNamespaceInfo(ext.getXmlDslModel().getNamespace(), ext.getXmlDslModel().getPrefix()))
          .collect(toImmutableList());
    } else {
      extensionNamespaces = of();
    }

    extensionsXmlInfoProvider = new StaticXmlNamespaceInfoProvider(extensionNamespaces);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected <T> Collection<T> doLookupProviders(Class<T> providerClass, ClassLoader classLoader) {
    Collection<T> providers = delegate.lookupProviders(providerClass, classLoader);
    if (XmlNamespaceInfoProvider.class.equals(providerClass)) {
      List<T> mutableProviders = new ArrayList<>(providers);
      mutableProviders.add((T) extensionsXmlInfoProvider);
      providers = copyOf(mutableProviders);
    }

    return providers;
  }

}
