/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.config.internal.resolvers;

import static org.mule.runtime.ast.graph.api.ArtifactAstDependencyGraphFactory.generateFor;
import static org.mule.test.allure.AllureConstants.LifecycleAndDependencyInjectionFeature.LIFECYCLE_AND_DEPENDENCY_INJECTION;
import static org.mule.test.allure.AllureConstants.LifecycleAndDependencyInjectionFeature.LifecyclePhaseStory.LIFECYCLE_PHASE_STORY;

import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsIterableWithSize.iterableWithSize;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.when;

import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.graph.api.ArtifactAstDependencyGraph;
import org.mule.runtime.ast.graph.api.ArtifactAstDependencyGraphFactory;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.util.Collection;
import java.util.Set;
import java.util.stream.Stream;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Issue("MULE-19984")
@Feature(LIFECYCLE_AND_DEPENDENCY_INJECTION)
@Story(LIFECYCLE_PHASE_STORY)
@ExtendWith(MockitoExtension.class)
class ConfigurationDependencyResolverTestCase extends AbstractMuleTestCase {

  private static final String COMPONENT_ID = "component";

  @Mock
  private ArtifactAst ast;

  @Mock
  private ArtifactAstDependencyGraph graph;

  @BeforeEach
  public void setUp() throws Exception {
    final var component = mock(ComponentAst.class);
    when(component.getComponentId()).thenReturn(of(COMPONENT_ID));
    when(ast.recursiveStream()).thenReturn(Stream.of(component));
  }

  @Test
  @Description("A set of two components will be returned when there are two required components")
  void getRequiredComponentsWithIdTestCase() {
    try (var graphFactoryStatic = mockStatic(ArtifactAstDependencyGraphFactory.class)) {
      graphFactoryStatic.when(() -> generateFor(ast)).thenReturn(graph);
      Set<ComponentAst> components = Set.of(createComponentWithId("A"),
                                            createComponentWithId("B"));
      when(graph.getRequiredComponents(COMPONENT_ID)).thenReturn(components);

      var configurationDependencyResolver = new ConfigurationDependencyResolver(ast);

      Collection<String> result = configurationDependencyResolver.getDirectComponentDependencies(COMPONENT_ID);
      assertThat(result, iterableWithSize(2));
    }
  }

  @Test
  @Description("A set of three components will be returned when there are 5 required components, only 3 have id")
  void getRequiredComponentsWithoutIdTestCase() {
    try (var graphFactoryStatic = mockStatic(ArtifactAstDependencyGraphFactory.class)) {
      graphFactoryStatic.when(() -> generateFor(ast)).thenReturn(graph);
      Set<ComponentAst> components = Set.of(createComponentWithId("A"),
                                            createComponentWithId("B"), createComponentWithId("C"),
                                            createComponentWithoutId("D"), createComponentWithoutId("E"));
      when(graph.getRequiredComponents(COMPONENT_ID)).thenReturn(components);

      var configurationDependencyResolver = new ConfigurationDependencyResolver(ast);

      Collection<String> result = configurationDependencyResolver.getDirectComponentDependencies(COMPONENT_ID);
      assertThat(result, iterableWithSize(3));
    }
  }

  private ComponentAst createComponentWithId(String componentName) {
    ComponentAst component = mock(ComponentAst.class);
    when(component.getComponentId()).thenReturn(of(componentName));
    return component;

  }

  private ComponentAst createComponentWithoutId(String componentName) {
    ComponentAst component = mock(ComponentAst.class);
    return component;
  }
}
