/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.config.internal.context.service;

import static org.mule.runtime.config.internal.context.service.InjectParamsFromContextServiceProxy.createInjectProviderParamsServiceProxy;
import static org.mule.runtime.config.internal.context.service.InjectParamsFromContextServiceUtils.MANY_CANDIDATES_ERROR_MSG_TEMPLATE;
import static org.mule.runtime.config.internal.context.service.InjectParamsFromContextServiceUtils.NO_OBJECT_FOUND_FOR_PARAM;
import static org.mule.runtime.config.utils.UtilsJavax.augmentedParam;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_MULE_CONTEXT;

import static java.lang.String.format;
import static java.lang.reflect.Proxy.newProxyInstance;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.Matchers.sameInstance;

import static org.junit.Assert.assertThrows;

import static org.mockito.Mockito.when;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.config.utils.UtilsJavax.AmbiguousAugmentedMethodService;
import org.mule.runtime.config.utils.UtilsJavax.AugmentedMethodService;
import org.mule.runtime.config.utils.UtilsJavax.AugmentedSubclassMethodService;
import org.mule.runtime.config.utils.UtilsJavax.AugmentedSubclassOverridesMethodService;
import org.mule.runtime.config.utils.UtilsJavax.BaseOverloadedService;
import org.mule.runtime.config.utils.UtilsJavax.BaseOverloadedService2;
import org.mule.runtime.config.utils.UtilsJavax.BaseService;
import org.mule.runtime.config.utils.UtilsJavax.BasicService;
import org.mule.runtime.config.utils.UtilsJavax.HiddenAugmentedMethodService;
import org.mule.runtime.config.utils.UtilsJavax.InvalidAugmentedMethodService;
import org.mule.runtime.config.utils.UtilsJavax.InvalidNamedAugmentedMethodService;
import org.mule.runtime.config.utils.UtilsJavax.NamedAugmentedMethodService;
import org.mule.runtime.config.utils.UtilsJavax.OverloadedAugmentedMethodService;
import org.mule.runtime.config.utils.UtilsJavax.OverloadedAugmentedMethodService2;
import org.mule.runtime.container.internal.MetadataInvocationHandler;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.registry.IllegalDependencyInjectionException;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;

import java.lang.reflect.Method;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

@SmallTest
@ExtendWith(MockitoExtension.class)
class InjectParamsFromContextServiceProxyJavaxTestCase extends AbstractMuleTestCase {

  @Mock
  private Registry registry;

  @Mock
  private MuleContext muleContext;

  @BeforeEach
  public void setUp() {
    augmentedParam = null;
  }

  @Test
  void notAugmentedInvocation() {
    BaseService service = new BasicService();

    final BaseService serviceProxy = (BaseService) createInjectProviderParamsServiceProxy(service, registry);

    serviceProxy.augmented();

    assertThat(augmentedParam, is(true));
  }

  @Test
  void augmentedInvocation() {
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseService service = new AugmentedMethodService();

    final BaseService serviceProxy = (BaseService) createInjectProviderParamsServiceProxy(service, registry);

    serviceProxy.augmented();

    assertThat(augmentedParam, sameInstance(muleContext));
  }

  @Test
  void augmentedSubclassInvocation() {
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseService service = new AugmentedSubclassMethodService();

    final BaseService serviceProxy = (BaseService) createInjectProviderParamsServiceProxy(service, registry);

    serviceProxy.augmented();

    assertThat(augmentedParam, sameInstance(muleContext));
  }

  @Test
  void augmentedSubclassOverridesInvocation() {
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseService service = new AugmentedSubclassOverridesMethodService();

    final BaseService serviceProxy = (BaseService) createInjectProviderParamsServiceProxy(service, registry);

    serviceProxy.augmented();

    assertThat(augmentedParam, is(true));
  }

  @Test
  void namedAugmentedInvocation() {
    when(registry.lookupByName(OBJECT_MULE_CONTEXT)).thenReturn(of(muleContext));
    BaseService service = new NamedAugmentedMethodService();

    final BaseService serviceProxy = (BaseService) createInjectProviderParamsServiceProxy(service, registry);

    serviceProxy.augmented();

    assertThat(augmentedParam, sameInstance(muleContext));
  }

  @Test
  void invalidNamedAugmentedInvocation() {
    BaseService service = new InvalidNamedAugmentedMethodService();

    final BaseService serviceProxy = (BaseService) createInjectProviderParamsServiceProxy(service, registry);

    var thrown = assertThrows(IllegalDependencyInjectionException.class,
                              () -> serviceProxy.augmented());
    assertThat(thrown.getMessage(),
               is(format(NO_OBJECT_FOUND_FOR_PARAM, "param", "augmented", "InvalidNamedAugmentedMethodService")));
  }

  @Test
  void hiddenAugmentedInvocation() {
    BaseService service = new HiddenAugmentedMethodService();

    final BaseService serviceProxy = (BaseService) createInjectProviderParamsServiceProxy(service, registry);

    serviceProxy.augmented();

    assertThat(augmentedParam, is(true));
  }

  @Test
  void overloadedAugmentedInvocation() {
    when(registry.lookupAllByType(int.class)).thenReturn(emptyList());
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseOverloadedService service = new OverloadedAugmentedMethodService();

    final BaseOverloadedService serviceProxy =
        (BaseOverloadedService) createInjectProviderParamsServiceProxy(service, registry);

    serviceProxy.augmented();

    assertThat(augmentedParam, is(true));
  }

  @Test
  void overloadedAugmentedInvocation2() {
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseOverloadedService service = new OverloadedAugmentedMethodService();

    final BaseOverloadedService serviceProxy =
        (BaseOverloadedService) createInjectProviderParamsServiceProxy(service, registry);

    serviceProxy.augmented(1);

    assertThat(augmentedParam, sameInstance(muleContext));
  }

  @Test
  void overloadedAugmentedInvocation3() {
    BaseOverloadedService2 service = new OverloadedAugmentedMethodService2();

    final BaseOverloadedService2 serviceProxy =
        (BaseOverloadedService2) createInjectProviderParamsServiceProxy(service, registry);

    serviceProxy.augmented(muleContext, 1);

    assertThat(augmentedParam, sameInstance(muleContext));
  }

  @Test
  void ambiguousAugmentedInvocation() {
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseService service = new AmbiguousAugmentedMethodService();

    final BaseService serviceProxy = (BaseService) createInjectProviderParamsServiceProxy(service, registry);

    var thrown = assertThrows(IllegalDependencyInjectionException.class,
                              () -> serviceProxy.augmented());
    assertThat(thrown.getMessage(),
               is(format(MANY_CANDIDATES_ERROR_MSG_TEMPLATE, "augmented", "AmbiguousAugmentedMethodService")));

    assertThat(augmentedParam, nullValue());
  }

  @Test
  void invalidAugmentedInvocation() {
    BaseService service = new InvalidAugmentedMethodService();

    final BaseService serviceProxy = (BaseService) createInjectProviderParamsServiceProxy(service, registry);

    serviceProxy.augmented();

    assertThat(augmentedParam, is(true));
  }

  @Test
  void throughProxyAugmentedInvocation() {
    when(registry.lookupAllByType(MuleContext.class)).thenReturn(singleton(muleContext));
    BaseService service = new AugmentedMethodService();

    final MetadataInvocationHandler<BaseService> noOpHandler = new MetadataInvocationHandler<>(service) {

      @Override
      public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        return method.invoke(service, args);
      }
    };
    final BaseService innerProxy =
        (BaseService) newProxyInstance(service.getClass().getClassLoader(), new Class<?>[] {BaseService.class}, noOpHandler);
    final BaseService serviceProxy = (BaseService) createInjectProviderParamsServiceProxy(innerProxy, registry);

    serviceProxy.augmented();

    assertThat(augmentedParam, sameInstance(muleContext));
  }

}
