/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.config.internal.context;

import static org.mule.runtime.api.config.FeatureFlaggingService.FEATURE_FLAGGING_SERVICE_KEY;
import static org.mule.runtime.api.serialization.ObjectSerializer.DEFAULT_OBJECT_SERIALIZER_NAME;
import static org.mule.runtime.api.time.TimeSupplier.OBJECT_TIME_SUPPLIER;
import static org.mule.runtime.config.internal.context.service.InjectParamsFromContextServiceProxy.createInjectProviderParamsServiceProxy;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_ARTIFACT_METER_PROVIDER_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_CORE_COMPONENT_TRACER_FACTORY_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_CORE_EVENT_TRACER_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_CORE_EXPORTER_FACTORY_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_CORE_SPAN_FACTORY_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_ERROR_METRICS_FACTORY_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_FILE_SPAN_EXPORTER_CONFIGURATION_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_MEMORY_MANAGEMENT_SERVICE;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_METER_EXPORTER_CONFIGURATION_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_METER_EXPORTER_FACTORY_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_METER_PROVIDER_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_PROFILING_SERVICE_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_SPAN_EXPORTER_CONFIGURATION_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_TRACER_INITIAL_SPAN_INFO_PROVIDER_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_TRACING_LEVEL_CONFIGURATION_KEY;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_ALERTING_SUPPORT;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_ARTIFACT_ENCODING;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_CONFIGURATION_PROPERTIES;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_DW_EXPRESSION_LANGUAGE_ADAPTER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_EXPRESSION_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_MULE_CONFIGURATION;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_REGISTRY;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_SCHEDULER_BASE_CONFIG;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_SCHEDULER_POOLS_CONFIG;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_STREAMING_GHOST_BUSTER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_STREAMING_MANAGER;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_TRANSFORMERS_REGISTRY;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_TRANSFORMER_RESOLVER;
import static org.mule.runtime.core.api.config.bootstrap.ArtifactType.DOMAIN;
import static org.mule.runtime.core.internal.config.bootstrap.AbstractRegistryBootstrap.BINDING_PROVIDER_PREDICATE;
import static org.mule.runtime.core.internal.config.bootstrap.AbstractRegistryBootstrap.TRANSFORMER_PREDICATE;
import static org.mule.runtime.core.internal.exception.ErrorTypeLocatorFactory.createDefaultErrorTypeLocator;
import static org.mule.runtime.metrics.exporter.api.MeterExporterProperties.METRIC_EXPORTER_ENABLED_PROPERTY;

import static java.lang.Boolean.getBoolean;
import static java.lang.reflect.Proxy.getInvocationHandler;
import static java.lang.reflect.Proxy.isProxyClass;
import static java.util.Map.entry;
import static java.util.Map.ofEntries;
import static java.util.Optional.of;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.api.component.location.ConfigurationComponentLocator;
import org.mule.runtime.api.config.FeatureFlaggingService;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.memory.management.MemoryManagementService;
import org.mule.runtime.api.scheduler.SchedulerContainerPoolsConfig;
import org.mule.runtime.api.service.Service;
import org.mule.runtime.config.internal.bean.DefaultObjectSerializerDelegate;
import org.mule.runtime.config.internal.bean.MuleConfigurationDelegate;
import org.mule.runtime.config.internal.context.service.InjectParamsFromContextServiceMethodInvoker;
import org.mule.runtime.config.internal.el.DataWeaveExtendedExpressionLanguageAdaptorFactoryBean;
import org.mule.runtime.config.internal.el.DefaultExpressionManagerFactoryBean;
import org.mule.runtime.config.internal.factories.SchedulerBaseConfigFactory;
import org.mule.runtime.config.internal.lazy.LazyDataWeaveExtendedExpressionLanguageAdaptorFactoryBean;
import org.mule.runtime.config.internal.registry.SpringRegistryBootstrap;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.runtime.core.api.management.stats.ArtifactMeterProvider;
import org.mule.runtime.core.internal.alert.DefaultAlertingSupport;
import org.mule.runtime.core.internal.config.CustomService;
import org.mule.runtime.core.internal.config.InternalCustomizationService;
import org.mule.runtime.core.internal.exception.ContributedErrorTypeLocator;
import org.mule.runtime.core.internal.exception.ContributedErrorTypeRepository;
import org.mule.runtime.core.internal.profiling.ProfilingServiceWrapper;
import org.mule.runtime.core.internal.registry.TypeBasedTransformerResolver;
import org.mule.runtime.core.internal.streaming.DefaultStreamingManager;
import org.mule.runtime.core.internal.streaming.StreamingGhostBuster;
import org.mule.runtime.core.internal.time.LocalTimeSupplier;
import org.mule.runtime.core.internal.transformer.DefaultTransformersRegistry;
import org.mule.runtime.core.privileged.exception.ErrorTypeLocator;
import org.mule.runtime.metrics.api.MeterProvider;
import org.mule.runtime.metrics.api.error.ErrorMetricsFactory;
import org.mule.runtime.metrics.exporter.impl.OpenTelemetryMeterExporterFactory;
import org.mule.runtime.metrics.exporter.impl.optel.config.OpenTelemetryAutoConfigurableMeterExporterConfiguration;
import org.mule.runtime.metrics.impl.DefaultMeterProvider;
import org.mule.runtime.metrics.impl.meter.error.DefaultErrorMetricsFactory;
import org.mule.runtime.module.service.internal.manager.LazyServiceProxy;
import org.mule.runtime.tracer.customization.impl.provider.DefaultInitialSpanInfoProvider;
import org.mule.runtime.tracer.exporter.config.impl.FileSpanExporterConfiguration;
import org.mule.runtime.tracer.exporter.impl.OpenTelemetrySpanExporterFactory;
import org.mule.runtime.tracer.exporter.impl.optel.config.OpenTelemetryAutoConfigurableSpanExporterConfiguration;
import org.mule.runtime.tracer.impl.CoreEventComponentTracerFactory;
import org.mule.runtime.tracer.impl.SelectableCoreEventTracer;
import org.mule.runtime.tracer.impl.span.factory.ExecutionSpanFactory;
import org.mule.runtime.tracing.level.impl.config.FileTracingLevelConfiguration;

import java.lang.reflect.InvocationHandler;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;

import jakarta.inject.Inject;

/**
 * This class configures the basic services available in a {@code MuleContext} that are independent of the artifact config.
 * <p>
 * There's a predefined set of services plus a configurable set of services provided by
 * {@code MuleContext#getCustomizationService}.
 * <p>
 * This class takes cares of registering bean definitions for each of the provided services so dependency injection can be
 * properly done through the use of {@link Inject}.
 *
 * @since 4.5
 */
public class BaseSpringMuleContextServiceConfigurator extends AbstractSpringMuleContextServiceConfigurator {

  private static final Logger LOGGER = LoggerFactory.getLogger(BaseSpringMuleContextServiceConfigurator.class);

  // This is needed just for some unit test scenarios
  // TODO MULE-20028 remove this
  @Deprecated
  public static final String DISABLE_TRANSFORMERS_SUPPORT =
      BaseSpringMuleContextServiceConfigurator.class.getName() + ".disableTransformersSupport";

  // Do not use static field. BeanDefinitions are reused and produce weird behaviour
  private final Map<String, BeanDefinition> baseContextServices =
      ofEntries(entry(OBJECT_TIME_SUPPLIER, getBeanDefinition(LocalTimeSupplier.class)),
                entry(OBJECT_ALERTING_SUPPORT, getBeanDefinition(DefaultAlertingSupport.class)),

                // OTEL Support
                entry(MULE_SPAN_EXPORTER_CONFIGURATION_KEY,
                      resolveSpanExporterConfiguration()),
                entry(MULE_FILE_SPAN_EXPORTER_CONFIGURATION_KEY,
                      resolveFileSpanExporterConfiguration()),
                entry(MULE_PROFILING_SERVICE_KEY, getBeanDefinition(ProfilingServiceWrapper.class)),
                entry(MULE_CORE_SPAN_FACTORY_KEY, getBeanDefinition(ExecutionSpanFactory.class)),
                entry(MULE_CORE_EXPORTER_FACTORY_KEY, getBeanDefinition(OpenTelemetrySpanExporterFactory.class)),
                entry(MULE_CORE_EVENT_TRACER_KEY, getBeanDefinition(SelectableCoreEventTracer.class)),
                entry(MULE_CORE_COMPONENT_TRACER_FACTORY_KEY, getBeanDefinition(CoreEventComponentTracerFactory.class)),
                entry(MULE_ERROR_METRICS_FACTORY_KEY, resolveErrorMetricsFactory()),
                entry(MULE_METER_PROVIDER_KEY, resolveBaseArtifactMeterProvider()),
                entry(MULE_METER_EXPORTER_CONFIGURATION_KEY,
                      getBeanDefinition(OpenTelemetryAutoConfigurableMeterExporterConfiguration.class)),
                entry(MULE_METER_EXPORTER_FACTORY_KEY, getBeanDefinition(OpenTelemetryMeterExporterFactory.class)),
                entry(MULE_TRACING_LEVEL_CONFIGURATION_KEY, getBeanDefinition(FileTracingLevelConfiguration.class)),
                entry(MULE_TRACER_INITIAL_SPAN_INFO_PROVIDER_KEY, getBeanDefinition(DefaultInitialSpanInfoProvider.class)),

                // Streaming
                entry(OBJECT_STREAMING_MANAGER, getBeanDefinition(DefaultStreamingManager.class)),
                entry(OBJECT_STREAMING_GHOST_BUSTER, getBeanDefinition(StreamingGhostBuster.class)));

  private final MuleContext muleContext;

  private final ArtifactType artifactType;
  private final ConfigurationProperties configurationProperties;
  private final boolean enableLazyInit;
  private final Registry serviceLocator;
  private final MemoryManagementService memoryManagementService;
  private org.mule.runtime.core.internal.registry.Registry originalRegistry;

  public BaseSpringMuleContextServiceConfigurator(MuleContext muleContext,
                                                  ConfigurationProperties configurationProperties,
                                                  ArtifactType artifactType,
                                                  BeanDefinitionRegistry beanDefinitionRegistry,
                                                  Registry serviceLocator,
                                                  MemoryManagementService memoryManagementService,
                                                  org.mule.runtime.core.internal.registry.Registry originalRegistry,
                                                  boolean enableLazyInit) {
    super((InternalCustomizationService) muleContext.getCustomizationService(), beanDefinitionRegistry);
    this.muleContext = muleContext;
    this.configurationProperties = configurationProperties;
    this.artifactType = artifactType;
    this.serviceLocator = serviceLocator;
    this.memoryManagementService = memoryManagementService;

    this.originalRegistry = originalRegistry;
    this.enableLazyInit = enableLazyInit;
  }

  void createArtifactServices() {
    FeatureFlaggingService featureFlaggingService = originalRegistry.lookupObject(FEATURE_FLAGGING_SERVICE_KEY);
    registerConstantBeanDefinition(FEATURE_FLAGGING_SERVICE_KEY, featureFlaggingService);
    registerConstantBeanDefinition(OBJECT_ARTIFACT_ENCODING, originalRegistry.lookupObject(OBJECT_ARTIFACT_ENCODING));

    registerConstantBeanDefinition(ConfigurationComponentLocator.REGISTRY_KEY, new BaseConfigurationComponentLocator());

    registerBeanDefinition(MULE_ARTIFACT_METER_PROVIDER_KEY, resolveArtifactIdMeterProvider());

    registerConstantBeanDefinition(MULE_MEMORY_MANAGEMENT_SERVICE, memoryManagementService);

    if (!artifactType.equals(DOMAIN)) {
      loadServiceConfigurators();
    }
    registerContextServices(baseContextServices, artifactType.equals(DOMAIN));

    // Instances of the repository and locator need to be injected into another objects before actually determining the possible
    // values. This contributing layer is needed to ensure the correct functioning of the DI mechanism while allowing actual
    // values to be provided at a later time.
    final ContributedErrorTypeRepository contributedErrorTypeRepository = new ContributedErrorTypeRepository();
    registerConstantBeanDefinition(ErrorTypeRepository.class.getName(), contributedErrorTypeRepository);
    final ContributedErrorTypeLocator contributedErrorTypeLocator = new ContributedErrorTypeLocator();
    contributedErrorTypeLocator
        .setDelegate(createDefaultErrorTypeLocator(contributedErrorTypeRepository, of(featureFlaggingService)));
    registerConstantBeanDefinition(ErrorTypeLocator.class.getName(), contributedErrorTypeLocator);

    registerConstantBeanDefinition(OBJECT_CONFIGURATION_PROPERTIES, configurationProperties);

    if (!getBoolean(DISABLE_TRANSFORMERS_SUPPORT)) {
      registerBeanDefinition(OBJECT_TRANSFORMER_RESOLVER, getBeanDefinition(TypeBasedTransformerResolver.class));
      registerBeanDefinition(OBJECT_TRANSFORMERS_REGISTRY, getBeanDefinition(DefaultTransformersRegistry.class));
    }

    registerLazyInitialisationAwareBeans();

    registerBeanDefinition(OBJECT_EXPRESSION_MANAGER, getBeanDefinition(DefaultExpressionManagerFactoryBean.class));

    registerBeanDefinition(OBJECT_SCHEDULER_POOLS_CONFIG,
                           getConstantObjectBeanDefinition(SchedulerContainerPoolsConfig.getInstance()));
    registerBeanDefinition(OBJECT_SCHEDULER_BASE_CONFIG, getBeanDefinition(SchedulerBaseConfigFactory.class));

    registerConstantBeanDefinition(OBJECT_REGISTRY, this.serviceLocator);
    registerBeanDefinition(DEFAULT_OBJECT_SERIALIZER_NAME, getBeanDefinition(DefaultObjectSerializerDelegate.class));
    registerBeanDefinition(OBJECT_MULE_CONFIGURATION, getBeanDefinition(MuleConfigurationDelegate.class));

    createRuntimeServices();
    createBootstrapBeanDefinitions();
    absorbOriginalRegistry();
  }

  private void registerLazyInitialisationAwareBeans() {
    if (enableLazyInit) {
      registerBeanDefinition(OBJECT_DW_EXPRESSION_LANGUAGE_ADAPTER,
                             getBeanDefinition(LazyDataWeaveExtendedExpressionLanguageAdaptorFactoryBean.class));
    } else {
      registerBeanDefinition(OBJECT_DW_EXPRESSION_LANGUAGE_ADAPTER,
                             getBeanDefinition(DataWeaveExtendedExpressionLanguageAdaptorFactoryBean.class));
    }
  }

  protected void createBootstrapBeanDefinitions() {
    try {
      SpringRegistryBootstrap springRegistryBootstrap =
          new SpringRegistryBootstrap(artifactType.getArtifactType(),
                                      muleContext.getRegistryBootstrapServiceDiscoverer(),
                                      this::registerBeanDefinition,
                                      BINDING_PROVIDER_PREDICATE
                                          .or(TRANSFORMER_PREDICATE));
      springRegistryBootstrap.initialise();
    } catch (InitialisationException e) {
      throw new RuntimeException(e);
    }
  }

  private void createRuntimeServices() {
    final Map<String, CustomService> customServices = getCustomizationService().getCustomServices();
    for (String serviceName : customServices.keySet()) {

      if (containsBeanDefinition(serviceName)) {
        throw new IllegalStateException("There is already a bean definition registered with key: " + serviceName);
      }

      final CustomService customService = customServices.get(serviceName);
      if (customService.isBaseContext()
          // TODO MULE-19927 get these form a more specific place and avoid this filter
          || isServiceRuntimeProvided(customService)) {
        final BeanDefinition beanDefinition = getCustomServiceBeanDefinition(customService, serviceName);

        LOGGER.debug("Registering runtime service '{}' for {}...", serviceName, artifactType.name());
        registerBeanDefinition(serviceName, beanDefinition);
      }
    }
  }

  @Override
  protected BeanDefinition addServiceProxyParamsInjector(Object servImpl) {
    BeanDefinition beanDefinition;
    if (isProxyClass(servImpl.getClass())) {
      InvocationHandler handler = getInvocationHandler(servImpl);
      if (handler instanceof LazyServiceProxy lazyHandker) {
        servImpl = lazyHandker
            .forApplication(new InjectParamsFromContextServiceMethodInvoker(serviceLocator));
      }

      beanDefinition = getConstantObjectBeanDefinition(servImpl, true);
    } else {
      beanDefinition =
          getConstantObjectBeanDefinition(createInjectProviderParamsServiceProxy((Service) servImpl, serviceLocator), true);
    }
    return beanDefinition;
  }

  private void absorbOriginalRegistry() {
    if (originalRegistry == null) {
      return;
    }

    originalRegistry.lookupByType(Object.class)
        .forEach(this::registerConstantBeanDefinition);
    originalRegistry = null;
  }

  private static BeanDefinition getPrimaryBeanDefinition(Object beanType) {
    BeanDefinition beanDefinition = getConstantObjectBeanDefinition(beanType);
    beanDefinition.setPrimary(true);

    return beanDefinition;
  }

  private BeanDefinition resolveBaseArtifactMeterProvider() {
    if (getBoolean(METRIC_EXPORTER_ENABLED_PROPERTY)) {
      return getBeanDefinition(DefaultMeterProvider.class);
    }
    return getConstantObjectBeanDefinition(MeterProvider.NO_OP);
  }

  private BeanDefinition resolveSpanExporterConfiguration() {
    return getBeanDefinitionBuilder(OpenTelemetryAutoConfigurableSpanExporterConfiguration.class)
        .addConstructorArgReference(MULE_FILE_SPAN_EXPORTER_CONFIGURATION_KEY).setPrimary(true).getBeanDefinition();
  }

  private BeanDefinition resolveFileSpanExporterConfiguration() {
    return getBeanDefinition(FileSpanExporterConfiguration.class);
  }

  private static BeanDefinition resolveErrorMetricsFactory() {
    if (getBoolean(METRIC_EXPORTER_ENABLED_PROPERTY)) {
      return getBeanDefinition(DefaultErrorMetricsFactory.class);
    }
    return getConstantObjectBeanDefinition(ErrorMetricsFactory.NO_OP);
  }

  private BeanDefinition resolveArtifactIdMeterProvider() {
    if (getBoolean(METRIC_EXPORTER_ENABLED_PROPERTY)) {
      return getBeanDefinitionBuilder(ArtifactMeterProvider.class)
          .addConstructorArgReference(MULE_METER_PROVIDER_KEY)
          .addConstructorArgValue(muleContext.getId()).setPrimary(true).getBeanDefinition();
    }
    return getPrimaryBeanDefinition(MeterProvider.NO_OP);
  }

}
