/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.config.internal.context;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.CONFIG;
import static org.mule.runtime.api.functional.Either.right;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.core.internal.management.stats.FlowClassifier.FlowType.APIKIT;
import static org.mule.runtime.core.internal.management.stats.FlowClassifier.FlowType.GENERIC;
import static org.mule.runtime.core.internal.management.stats.FlowClassifier.FlowType.SOAPKIT;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.functional.Either;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.core.internal.logger.CustomLogger;
import org.mule.runtime.core.internal.management.stats.FlowClassifier;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.util.List;

import io.qameta.allure.Description;
import io.qameta.allure.Issue;
import org.hamcrest.Matcher;
import org.hamcrest.collection.IsEmptyCollection;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.helpers.SubstituteLogger;

public class FlowClassifierFactoryTestCase extends AbstractMuleTestCase {

  private static final String APIKIT_EXTENSION_NAME = "APIKit";
  private static final String SOAPKIT_EXTENSION_NAME = "APIKit for SOAP";

  private CustomLogger logger;
  private Logger originalDelegate;
  private ArtifactAst artifactAst;
  private ArtifactAst parentArtifactAst;
  private ComponentAst apiKitConfig;
  private ComponentAst soapKitConfig;
  private ComponentAst parentApiKitConfig;
  private ComponentAst parentSoapKitConfig;

  @Before
  public void setupLogger() {
    SubstituteLogger substituteLogger = (SubstituteLogger) LoggerFactory.getLogger(FlowClassifierFactory.class);
    this.originalDelegate = substituteLogger.delegate();
    this.logger = new CustomLogger(originalDelegate, FlowClassifierFactory.class.getName());
    substituteLogger.setDelegate(this.logger);
  }

  @After
  public void restoreLogger() {
    SubstituteLogger substituteLogger = (SubstituteLogger) LoggerFactory.getLogger(FlowClassifierFactory.class);
    substituteLogger.setDelegate(this.originalDelegate);
  }

  @Before
  public void setUp() {
    artifactAst = mock(ArtifactAst.class);
    parentArtifactAst = mock(ArtifactAst.class);
    apiKitConfig = mock(ComponentAst.class);
    soapKitConfig = mock(ComponentAst.class);
    parentApiKitConfig = mock(ComponentAst.class);
    parentSoapKitConfig = mock(ComponentAst.class);

    setupConfigMock(apiKitConfig, "api-config", APIKIT_EXTENSION_NAME);
    setupConfigMock(soapKitConfig, "soapkit-config", SOAPKIT_EXTENSION_NAME);
    setupConfigMock(parentApiKitConfig, "parent-api-config", APIKIT_EXTENSION_NAME);
    setupConfigMock(parentSoapKitConfig, "parent-soapkit-config", SOAPKIT_EXTENSION_NAME);

    // Setup APIKit config with flow mappings
    ComponentAst flowMapping = createFlowMappingMock(right("mapped-flow"));
    setupFlowMappingsParameter(apiKitConfig, right(singletonList(flowMapping)));

    // Setup parent APIKit config with flow mappings
    ComponentAst parentFlowMapping = createFlowMappingMock(right("parent-mapped-flow"));
    setupFlowMappingsParameter(parentApiKitConfig, right(singletonList(parentFlowMapping)));
  }

  private FlowClassifier createClassifier(List<ComponentAst> components, List<ComponentAst> parentComponents) {
    when(artifactAst.topLevelComponentsStream()).thenAnswer(invocation -> components.stream());
    when(artifactAst.getParent()).thenReturn(of(parentArtifactAst));
    when(parentArtifactAst.topLevelComponentsStream()).thenAnswer(invocation -> parentComponents.stream());
    return new FlowClassifierFactory(artifactAst).create();
  }

  @Test
  public void testGenericFlow() {
    FlowClassifier classifier = createClassifier(emptyList(), emptyList());
    assertThat("generic-flow", classifier.getFlowType("generic-flow"), is(GENERIC));
  }

  @Test
  public void testApiKitFlow() {
    FlowClassifier classifier = createClassifier(singletonList(apiKitConfig), emptyList());
    assertThat("mapped-flow", classifier.getFlowType("mapped-flow"), is(APIKIT));
    assertThat("flow:api-config", classifier.getFlowType("flow:api-config"), is(APIKIT));
  }

  @Test
  public void testSoapKitFlow() {
    FlowClassifier classifier = createClassifier(singletonList(soapKitConfig), emptyList());
    assertThat("flow:\\soapkit-config", classifier.getFlowType("flow:\\soapkit-config"), is(SOAPKIT));
  }

  @Test
  public void testMultipleConfigs() {
    FlowClassifier classifier = createClassifier(asList(apiKitConfig, soapKitConfig), emptyList());
    assertThat("mapped-flow", classifier.getFlowType("mapped-flow"), is(APIKIT));
    assertThat("flow:api-config", classifier.getFlowType("flow:api-config"), is(APIKIT));
    assertThat("flow:\\soapkit-config", classifier.getFlowType("flow:\\soapkit-config"), is(SOAPKIT));
    assertThat("generic-flow", classifier.getFlowType("generic-flow"), is(GENERIC));
  }

  @Test
  public void testApiKitFlowWithNonMatchingConfig() {
    FlowClassifier classifier = createClassifier(singletonList(apiKitConfig), emptyList());
    assertThat("flow:non-matching-config", classifier.getFlowType("flow:non-matching-config"), is(GENERIC));
  }

  @Test
  public void testSoapKitFlowWithNonMatchingConfig() {
    FlowClassifier classifier = createClassifier(singletonList(soapKitConfig), emptyList());
    assertThat("flow:\\non-matching-config", classifier.getFlowType("flow:\\non-matching-config"), is(GENERIC));
  }

  @Test
  public void testMultipleConfigsWithNonMatchingFlows() {
    FlowClassifier classifier = createClassifier(asList(apiKitConfig, soapKitConfig), emptyList());
    assertThat("flow:non-matching-api-config", classifier.getFlowType("flow:non-matching-api-config"), is(GENERIC));
    assertThat("flow:\\non-matching-soap-config", classifier.getFlowType("flow:\\non-matching-soap-config"), is(GENERIC));
  }

  @Test
  public void testApiKitFlowFromParent() {
    FlowClassifier classifier = createClassifier(emptyList(), singletonList(parentApiKitConfig));
    assertThat("parent-mapped-flow", classifier.getFlowType("parent-mapped-flow"), is(APIKIT));
    assertThat("flow:parent-api-config", classifier.getFlowType("flow:parent-api-config"), is(APIKIT));
  }

  @Test
  public void testSoapKitFlowFromParent() {
    FlowClassifier classifier = createClassifier(emptyList(), singletonList(parentSoapKitConfig));
    assertThat("flow:\\parent-soapkit-config", classifier.getFlowType("flow:\\parent-soapkit-config"), is(SOAPKIT));
  }

  @Test
  public void testMultipleConfigsFromParent() {
    FlowClassifier classifier = createClassifier(emptyList(), asList(parentApiKitConfig, parentSoapKitConfig));
    assertThat("parent-mapped-flow", classifier.getFlowType("parent-mapped-flow"), is(APIKIT));
    assertThat("flow:parent-api-config", classifier.getFlowType("flow:parent-api-config"), is(APIKIT));
    assertThat("flow:\\parent-soapkit-config", classifier.getFlowType("flow:\\parent-soapkit-config"), is(SOAPKIT));
    assertThat("generic-flow", classifier.getFlowType("generic-flow"), is(GENERIC));
  }

  @Test
  public void testConfigsFromBothMainAndParent() {
    FlowClassifier classifier = createClassifier(asList(apiKitConfig, soapKitConfig),
                                                 asList(parentApiKitConfig, parentSoapKitConfig));
    assertThat("mapped-flow", classifier.getFlowType("mapped-flow"), is(APIKIT));
    assertThat("flow:api-config", classifier.getFlowType("flow:api-config"), is(APIKIT));
    assertThat("flow:\\soapkit-config", classifier.getFlowType("flow:\\soapkit-config"), is(SOAPKIT));
    assertThat("parent-mapped-flow", classifier.getFlowType("parent-mapped-flow"), is(APIKIT));
    assertThat("flow:parent-api-config", classifier.getFlowType("flow:parent-api-config"), is(APIKIT));
    assertThat("flow:\\parent-soapkit-config", classifier.getFlowType("flow:\\parent-soapkit-config"), is(SOAPKIT));
    assertThat("generic-flow", classifier.getFlowType("generic-flow"), is(GENERIC));
  }

  @Test
  @Issue("W-19004884")
  @Description("The ApiKit config doesn't have any value for the optional flow-mappings parameter, it must not fail nor log")
  public void testApiKitNoFlowMappingsParameterPresent() {
    artifactAst = mock(ArtifactAst.class);
    apiKitConfig = mock(ComponentAst.class);
    setupConfigMock(apiKitConfig, "api-config", APIKIT_EXTENSION_NAME);
    setupFlowMappingsParameter(apiKitConfig, Either.empty());

    FlowClassifier classifier = createClassifier(singletonList(apiKitConfig), emptyList());
    assertThat("mapped-flow", classifier.getFlowType("mapped-flow"), is(GENERIC));
    assertThat("flow:api-config", classifier.getFlowType("flow:api-config"), is(APIKIT));
    assertOnLog(IsEmptyCollection.empty());
  }

  @Test
  @Issue("W-19004884")
  @Description("In this test, the ApiKit config model doesn't even have the flow-mappings parameter, something changed in the model, it must not fail, but log")
  public void testApiKitNoFlowMappingsParameter() {
    artifactAst = mock(ArtifactAst.class);
    apiKitConfig = mock(ComponentAst.class);
    setupConfigMock(apiKitConfig, "api-config", APIKIT_EXTENSION_NAME);

    FlowClassifier classifier = createClassifier(singletonList(apiKitConfig), emptyList());
    assertThat("mapped-flow", classifier.getFlowType("mapped-flow"), is(GENERIC));
    assertThat("flow:api-config", classifier.getFlowType("flow:api-config"), is(APIKIT));
    assertOnLog(containsInAnyOrder("Component `api-config` is a config from `APIKit` but has no flow-mappings parameter, the model must have changed incompatibly"));
  }

  @Test
  @Issue("W-19004884")
  @Description("The ApiKit config does have a value for the optional flow-mappings parameter but it is not iterable, it must not fail, but log")
  public void testApiKitFlowMappingsNotIterable() {
    artifactAst = mock(ArtifactAst.class);
    apiKitConfig = mock(ComponentAst.class);
    setupConfigMock(apiKitConfig, "api-config", APIKIT_EXTENSION_NAME);
    setupFlowMappingsParameter(apiKitConfig, right("some unexpected value"));

    FlowClassifier classifier = createClassifier(singletonList(apiKitConfig), emptyList());
    assertThat("mapped-flow", classifier.getFlowType("mapped-flow"), is(GENERIC));
    assertThat("flow:api-config", classifier.getFlowType("flow:api-config"), is(APIKIT));
    assertOnLog(containsInAnyOrder("Component `api-config` has flow mappings but they are not iterable, the model `APIKit` must have changed incompatibly"));
  }

  @Test
  @Issue("W-19004884")
  @Description("The ApiKit config does have a value for the optional flow-mappings parameter but it is not an iterable of ComponentAst, it must not fail, but log")
  public void testApiKitFlowMappingsIterableButNotComponentAst() {
    artifactAst = mock(ArtifactAst.class);
    apiKitConfig = mock(ComponentAst.class);
    setupConfigMock(apiKitConfig, "api-config", APIKIT_EXTENSION_NAME);
    setupFlowMappingsParameter(apiKitConfig, right(asList(null, "some unexpected value")));

    FlowClassifier classifier = createClassifier(singletonList(apiKitConfig), emptyList());
    assertThat("mapped-flow", classifier.getFlowType("mapped-flow"), is(GENERIC));
    assertThat("flow:api-config", classifier.getFlowType("flow:api-config"), is(APIKIT));

    // We expect the message to be logged twice in this test because there are two invalid items
    Matcher<String> matcher =
        is("Component `api-config` has a flow-mapping that is not of the expected type, the model `APIKit` must have changed incompatibly");
    assertOnLog(containsInAnyOrder(matcher, matcher));
  }

  @Test
  @Issue("W-19004884")
  @Description("Some flow-mappings in the ApiKit config don't have a value for the required parameter `flow-ref`, it must not fail nor log")
  public void testApiKitFlowMappingsNoFlowRef() {
    artifactAst = mock(ArtifactAst.class);
    apiKitConfig = mock(ComponentAst.class);
    setupConfigMock(apiKitConfig, "api-config", APIKIT_EXTENSION_NAME);
    ComponentAst flowMapping = createFlowMappingMock(Either.empty());
    setupFlowMappingsParameter(apiKitConfig, right(singletonList(flowMapping)));

    FlowClassifier classifier = createClassifier(singletonList(apiKitConfig), emptyList());
    assertThat("mapped-flow", classifier.getFlowType("mapped-flow"), is(GENERIC));
    assertThat("flow:api-config", classifier.getFlowType("flow:api-config"), is(APIKIT));
    assertOnLog(containsInAnyOrder("Component `api-config` has a flow-mapping with no flow-ref which is supposed to be a required parameter, the model `APIKit` must have changed incompatibly"));
  }

  @Test
  @Issue("W-19004884")
  @Description("Some flow-mappings in the ApiKit config have a `flow-ref` that is not a String, it must not fail, but log")
  public void testApiKitFlowMappingsInvalidFlowRefType() {
    artifactAst = mock(ArtifactAst.class);
    apiKitConfig = mock(ComponentAst.class);
    setupConfigMock(apiKitConfig, "api-config", APIKIT_EXTENSION_NAME);
    ComponentAst flowMapping = createFlowMappingMock(right(1));
    setupFlowMappingsParameter(apiKitConfig, right(singletonList(flowMapping)));

    FlowClassifier classifier = createClassifier(singletonList(apiKitConfig), emptyList());
    assertThat("mapped-flow", classifier.getFlowType("mapped-flow"), is(GENERIC));
    assertThat("flow:api-config", classifier.getFlowType("flow:api-config"), is(APIKIT));
    assertOnLog(containsInAnyOrder("Component `api-config` has a flow-mapping with a flow-ref that is not a String value, the model `APIKit` must have changed incompatibly"));
  }

  private void setupConfigMock(ComponentAst config, String configId, String extensionName) {
    when(config.getComponentType()).thenReturn(CONFIG);
    when(config.getComponentId()).thenReturn(of(configId));
    ExtensionModel extensionModel = mock(ExtensionModel.class);
    when(extensionModel.getName()).thenReturn(extensionName);
    when(config.getExtensionModel()).thenReturn(extensionModel);
    ConfigurationModel configModel = mock(ConfigurationModel.class);
    when(configModel.getName()).thenReturn("config");
    when(config.getModel(ConfigurationModel.class)).thenReturn(of(configModel));
  }

  private void setupFlowMappingsParameter(ComponentAst config, Either<String, Object> flowMappingsParameterValue) {
    ComponentParameterAst flowMappingsParameter = mock(ComponentParameterAst.class);
    when(flowMappingsParameter.getValue()).thenReturn(flowMappingsParameterValue);
    when(config.getParameter(DEFAULT_GROUP_NAME, "flowMappings")).thenReturn(flowMappingsParameter);
  }

  private ComponentAst createFlowMappingMock(Either<String, Object> flowMappingsParameterValue) {
    ComponentAst flowMapping = mock(ComponentAst.class);
    ComponentParameterAst parentFlowRefParameter = mock(ComponentParameterAst.class);
    when(parentFlowRefParameter.getValue()).thenReturn(flowMappingsParameterValue);
    when(flowMapping.getParameter("FlowMapping", "flow-ref")).thenReturn(parentFlowRefParameter);
    return flowMapping;
  }

  private void assertOnLog(Matcher<? super List<String>> matcher) {
    assertThat(logger.getMessages(), matcher);
  }
}
