/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.tls.internal.socket;

import static org.mule.runtime.api.util.MuleSystemProperties.ENABLE_TLS_CONF_ARTIFACT_CLASSLOADER_LOOKUP;
import static org.mule.runtime.core.internal.util.PropertiesUtils.loadProperties;

import static java.lang.Boolean.getBoolean;
import static java.lang.Thread.currentThread;

import static org.apache.commons.lang3.StringUtils.split;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Properties;

import org.apache.commons.lang3.stream.Streams;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TlsProperties {

  private static final Logger logger = LoggerFactory.getLogger(TlsProperties.class);

  private String[] enabledCipherSuites;
  private String[] enabledProtocols;
  private String defaultProtocol;

  public String[] getEnabledCipherSuites() {
    return enabledCipherSuites;
  }

  public String[] getEnabledProtocols() {
    return enabledProtocols;
  }

  public String getDefaultProtocol() {
    return defaultProtocol;
  }

  public void load(String fileName) {
    try (InputStream config = resolveConfigStream(fileName)) {
      if (config == null) {
        logger.warn("File {} not found, using default configuration.", fileName);
      } else {
        logger.info("Loading configuration file: {}", fileName);
        Properties properties = loadProperties(config);

        String enabledCipherSuitesProperty = properties.getProperty("enabledCipherSuites");
        String enabledProtocolsProperty = properties.getProperty("enabledProtocols");
        String defaultProtocolProperty = properties.getProperty("defaultProtocol");

        if (enabledCipherSuitesProperty != null) {
          enabledCipherSuites = Streams.of(split(enabledCipherSuitesProperty, ",")).map(String::trim).toArray(String[]::new);

        }
        if (enabledProtocolsProperty != null) {
          enabledProtocols = Streams.of(split(enabledProtocolsProperty, ",")).map(String::trim).toArray(String[]::new);
        }
        if (defaultProtocolProperty != null) {
          defaultProtocol = defaultProtocolProperty.trim();
        }
      }
    } catch (IOException e) {
      logger.atWarn()
          .setCause(e)
          .log("Cannot read file {}, using default configuration", fileName);
    }
  }

  private InputStream resolveConfigStream(String fileName) {
    InputStream configStream = null;
    if (allowTlsConfArtifactClassLoaderLookup()) {
      configStream = currentThread().getContextClassLoader().getResourceAsStream(fileName);
    }

    if (configStream == null) {
      configStream = TlsProperties.class.getClassLoader().getResourceAsStream(fileName);
    }

    return configStream;
  }

  private boolean allowTlsConfArtifactClassLoaderLookup() {
    return getBoolean(ENABLE_TLS_CONF_ARTIFACT_CLASSLOADER_LOOKUP);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    TlsProperties that = (TlsProperties) o;

    if (!Arrays.equals(enabledCipherSuites, that.enabledCipherSuites)) {
      return false;
    }
    if (!Arrays.equals(enabledProtocols, that.enabledProtocols)) {
      return false;
    }

    return defaultProtocol != null ? defaultProtocol.equals(that.defaultProtocol) : that.defaultProtocol == null;
  }

  @Override
  public int hashCode() {
    int result = enabledCipherSuites != null ? Arrays.hashCode(enabledCipherSuites) : 0;
    result = 31 * result + (enabledProtocols != null ? Arrays.hashCode(enabledProtocols) : 0);
    result = 31 * result + (defaultProtocol != null ? defaultProtocol.hashCode() : 0);
    return result;
  }
}
