/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.test.module.tls;

import static org.mule.runtime.api.util.MuleSystemProperties.ENABLE_TLS_STORES_FILESYSTEM_LOOKUP;
import static org.mule.runtime.api.util.MuleSystemProperties.MULE_SECURITY_SYSTEM_PROPERTY;
import static org.mule.runtime.core.internal.test.util.TestFileUtils.isFileOpen;
import static org.mule.runtime.module.tls.internal.TlsConfiguration.DEFAULT_KEYSTORE;
import static org.mule.runtime.module.tls.internal.TlsConfiguration.DEFAULT_SECURITY_MODEL;
import static org.mule.runtime.module.tls.internal.TlsConfiguration.DEFAULT_SSL_TYPE;
import static org.mule.runtime.module.tls.internal.TlsConfiguration.JSSE_NAMESPACE;
import static org.mule.runtime.module.tls.internal.TlsConfiguration.PROPERTIES_FILE_PATTERN;
import static org.mule.test.allure.AllureConstants.TlsSsl.TLS_SSL_FEATURE;
import static org.mule.test.allure.AllureConstants.TlsSsl.TlsSslStory.STORES_VISIBILITY_STORY;

import static java.lang.String.format;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.arrayContainingInAnyOrder;
import static org.hamcrest.Matchers.arrayWithSize;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.io.FileMatchers.anExistingFile;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertThrows;
import static org.junit.Assert.assertTrue;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.module.tls.internal.TlsConfiguration;
import org.mule.runtime.module.tls.internal.util.SecurityUtils;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.URISyntaxException;
import java.net.URL;
import java.security.CodeSource;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLServerSocket;
import javax.net.ssl.SSLServerSocketFactory;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;

import org.junit.jupiter.api.Test;
import org.junitpioneer.jupiter.SetSystemProperty;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(TLS_SSL_FEATURE)
public class TlsConfigurationTestCase extends AbstractMuleTestCase {

  private static final String SUPPORTED_CIPHER_SUITE = "TLS_DHE_DSS_WITH_AES_128_CBC_SHA";
  private static final String SUPPORTED_PROTOCOL = "TLSv1.1";
  private static final String TEST_SECURITY_MODEL = "test";

  @Test
  void testEmptyConfiguration() throws Exception {
    TlsConfiguration configuration = new TlsConfiguration(DEFAULT_KEYSTORE);
    assertThrows("no key password", NullPointerException.class, () -> configuration.initialise(false, JSSE_NAMESPACE));

    configuration.setKeyPassword("mulepassword");
    assertThrows("no store password", NullPointerException.class, () -> configuration.initialise(false, JSSE_NAMESPACE));

    configuration.setKeyStorePassword("mulepassword");
    configuration.setKeyStore(""); // guaranteed to not exist
    assertThrows("no keystore", CreateException.class, () -> configuration.initialise(false, JSSE_NAMESPACE));
  }

  @Test
  void testTlsConfigurationDoesNotLeakKeyStoreFile() throws Exception {
    TlsConfiguration configuration = new TlsConfiguration(DEFAULT_KEYSTORE);
    configuration.setKeyPassword("mulepassword");
    configuration.setKeyStorePassword("mulepassword");
    configuration.setKeyStore("clientKeystore");
    configuration.initialise(false, JSSE_NAMESPACE);

    File keyStoreFile = new File(this.getClass().getClassLoader().getResource(configuration.getKeyStore()).toURI());
    assertThat(keyStoreFile, anExistingFile());
    assertThat(isFileOpen(keyStoreFile), is(false));
  }

  @Test
  void testTlsConfigurationDoesNotLeakTrustStoreFile() throws Exception {
    TlsConfiguration configuration = new TlsConfiguration(DEFAULT_KEYSTORE);
    configuration.setKeyPassword("mulepassword");
    configuration.setKeyStorePassword("mulepassword");
    configuration.setKeyStore("clientKeystore");
    configuration.setTrustStorePassword("mulepassword");
    configuration.setTrustStore("trustStore");
    configuration.initialise(false, JSSE_NAMESPACE);

    File trustStoreFile = new File(this.getClass().getClassLoader().getResource(configuration.getTrustStore()).toURI());
    assertThat(trustStoreFile, anExistingFile());
    assertThat(isFileOpen(trustStoreFile), is(false));
  }

  @Test
  @Issue("MULE-18569")
  @Description("When store file doesn't exist, the absolute path is null")
  void setNotExistentPathLetsNullValue() throws IOException {
    TlsConfiguration configuration = new TlsConfiguration(DEFAULT_KEYSTORE);
    configuration.setKeyStore("notExistent");
    configuration.setTrustStore("notExistent");

    assertThat(configuration.getKeyStore(), is(nullValue()));
    assertThat(configuration.getTrustStore(), is(nullValue()));
  }

  @Test
  @SetSystemProperty(key = ENABLE_TLS_STORES_FILESYSTEM_LOOKUP, value = "true")
  @Issue("MULE-18569")
  @Description("The TLS Configuration path setters were prepending a slash to the absolute path in Windows")
  void tlsConfigurationDoesNotBreakPaths() throws IOException, URISyntaxException {
    TlsConfiguration configuration = new TlsConfiguration(DEFAULT_KEYSTORE);

    configuration.setKeyStore(resourceToFile("clientKeystore"));
    configuration.setTrustStore(resourceToFile("trustStore"));

    assertThat(new File(configuration.getKeyStore()), anExistingFile());
    assertThat(new File(configuration.getTrustStore()), anExistingFile());
  }

  @Test
  void testSimpleSocket() throws Exception {
    TlsConfiguration configuration = new TlsConfiguration(DEFAULT_KEYSTORE);
    configuration.setKeyPassword("mulepassword");
    configuration.setKeyStorePassword("mulepassword");
    configuration.setKeyStore("clientKeystore");
    configuration.initialise(false, JSSE_NAMESPACE);
    SSLSocketFactory socketFactory = configuration.getSocketFactory();
    assertTrue("socket is useless", socketFactory.getSupportedCipherSuites().length > 0);
  }

  @Test
  void testTlsConfigurationUsingPKCS12KeystoreWithMultipleKeys() throws Exception {
    TlsConfiguration configuration = new TlsConfiguration(DEFAULT_KEYSTORE);
    configuration.setKeyPassword("passw0rd");
    configuration.setKeyStorePassword("passw0rd");
    configuration.setKeyStore("keystoreMultiplesKeys.p12");
    configuration.setKeyStoreType("pkcs12");
    configuration.setKeyAlias("ldnmulvs01");
    configuration.initialise(false, JSSE_NAMESPACE);
    SSLSocketFactory socketFactory = configuration.getSocketFactory();
    assertTrue("socket is useless", socketFactory.getSupportedCipherSuites().length > 0);
  }

  @Test
  void testExceptionOnInvalidKeyAlias() throws Exception {
    TlsConfiguration config = new TlsConfiguration("serverKeystore");
    config.setKeyStorePassword("mulepassword");
    config.setKeyPassword("mulepassword");
    config.setKeyAlias("this_key_does_not_exist_in_the_keystore");

    var thrown = assertThrows(CreateException.class, () -> config.initialise(false, JSSE_NAMESPACE));
    assertThat(thrown.getCause(), instanceOf(IllegalStateException.class));
  }


  @Test
  void testCipherSuitesFromConfigFile() throws Exception {
    File configFile = createDefaultConfigFile();

    try {
      TlsConfiguration tlsConfiguration = new TlsConfiguration(DEFAULT_KEYSTORE);
      tlsConfiguration.initialise(true, JSSE_NAMESPACE);

      SSLSocket socket = (SSLSocket) tlsConfiguration.getSocketFactory().createSocket();
      SSLServerSocket serverSocket = (SSLServerSocket) tlsConfiguration.getServerSocketFactory().createServerSocket();

      assertArrayEquals(new String[] {SUPPORTED_CIPHER_SUITE}, socket.getEnabledCipherSuites());
      assertArrayEquals(new String[] {SUPPORTED_CIPHER_SUITE}, serverSocket.getEnabledCipherSuites());
    } finally {
      configFile.delete();
    }
  }

  @Test
  void testProtocolsFromConfigFile() throws Exception {
    File configFile = createDefaultConfigFile();

    try {
      TlsConfiguration tlsConfiguration = new TlsConfiguration(DEFAULT_KEYSTORE);
      tlsConfiguration.initialise(true, JSSE_NAMESPACE);

      SSLSocket socket = (SSLSocket) tlsConfiguration.getSocketFactory().createSocket();
      SSLServerSocket serverSocket = (SSLServerSocket) tlsConfiguration.getServerSocketFactory().createServerSocket();

      assertArrayEquals(new String[] {SUPPORTED_PROTOCOL}, socket.getEnabledProtocols());
      assertArrayEquals(new String[] {SUPPORTED_PROTOCOL}, serverSocket.getEnabledProtocols());
    } finally {
      configFile.delete();
    }
  }

  @Test
  void defaultProtocol() throws Exception {
    TlsConfiguration tlsConfiguration = new TlsConfiguration(DEFAULT_KEYSTORE);
    tlsConfiguration.initialise(true, JSSE_NAMESPACE);

    SSLSocketFactory socketFactory = tlsConfiguration.getSocketFactory();
    SSLServerSocketFactory serverSocketFactory = tlsConfiguration.getServerSocketFactory();

    SSLContext sslContext = SSLContext.getInstance(DEFAULT_SSL_TYPE);
    sslContext.init(null, null, null);

    assertThat(socketFactory.getDefaultCipherSuites(),
               arrayContainingInAnyOrder(sslContext.getSocketFactory().getDefaultCipherSuites()));
  }

  @Test
  void defaultProtocolFromConfigFile() throws Exception {
    File configFile = createDefaultProtocolConfigFile();

    try {
      TlsConfiguration tlsConfiguration = new TlsConfiguration(DEFAULT_KEYSTORE);
      tlsConfiguration.initialise(true, JSSE_NAMESPACE);

      SSLSocketFactory socketFactory = tlsConfiguration.getSocketFactory();
      SSLServerSocketFactory serverSocketFactory = tlsConfiguration.getServerSocketFactory();

      SSLContext sslContext = SSLContext.getInstance(SUPPORTED_PROTOCOL);
      sslContext.init(null, null, null);

      SSLSocketFactory protocolSocketFactory = sslContext.getSocketFactory();
      SSLServerSocketFactory protocolServerSocketFactory = sslContext.getServerSocketFactory();

      assertThat(socketFactory.getDefaultCipherSuites(), arrayWithSize(protocolSocketFactory.getDefaultCipherSuites().length));
      assertThat(socketFactory.getDefaultCipherSuites(),
                 is(arrayContainingInAnyOrder(protocolSocketFactory.getDefaultCipherSuites())));
      assertThat(serverSocketFactory.getDefaultCipherSuites(),
                 arrayWithSize(protocolServerSocketFactory.getDefaultCipherSuites().length));
      assertThat(serverSocketFactory.getDefaultCipherSuites(),
                 is(arrayContainingInAnyOrder(protocolServerSocketFactory.getDefaultCipherSuites())));
    } finally {
      configFile.delete();
    }
  }

  @Test
  void overrideDefaultProtocolFromConfigFile() throws Exception {
    File configFile = createDefaultProtocolConfigFile();

    try {
      TlsConfiguration tlsConfiguration = new TlsConfiguration(DEFAULT_KEYSTORE);
      tlsConfiguration.setSslType("TLSv1.2");
      tlsConfiguration.initialise(true, JSSE_NAMESPACE);

      SSLSocketFactory socketFactory = tlsConfiguration.getSocketFactory();

      SSLContext sslContext = SSLContext.getInstance(SUPPORTED_PROTOCOL);
      sslContext.init(null, null, null);

      SSLSocketFactory protocolSocketFactory = sslContext.getSocketFactory();

      assertThat(socketFactory.getDefaultCipherSuites(),
                 not(arrayWithSize(protocolSocketFactory.getDefaultCipherSuites().length)));
    } finally {
      configFile.delete();
    }
  }

  @Test
  void testSecurityModelProperty() throws Exception {
    String previousSecurityModel = SecurityUtils.getSecurityModel();
    System.setProperty(MULE_SECURITY_SYSTEM_PROPERTY, TEST_SECURITY_MODEL);
    File file = createConfigFile(TEST_SECURITY_MODEL, "enabledCipherSuites=TEST");

    try {
      TlsConfiguration tlsConfiguration = new TlsConfiguration(DEFAULT_KEYSTORE);
      tlsConfiguration.initialise(true, JSSE_NAMESPACE);

      assertArrayEquals(new String[] {"TEST"}, tlsConfiguration.getEnabledCipherSuites());
    } finally {
      System.setProperty(MULE_SECURITY_SYSTEM_PROPERTY, previousSecurityModel);
      file.delete();
    }
  }

  @Test
  @Story(STORES_VISIBILITY_STORY)
  void trustStoreNotReadFromFilesystem() throws IOException, URISyntaxException {
    TlsConfiguration configuration = new TlsConfiguration(DEFAULT_KEYSTORE);

    configuration.setTrustStore(resourceToFile("trustStore"));
    assertThat(configuration.getTrustStore(), nullValue());
  }

  @Test
  @Story(STORES_VISIBILITY_STORY)
  void keyStoreNotReadFromFilesystem() throws IOException, URISyntaxException {
    TlsConfiguration configuration = new TlsConfiguration(DEFAULT_KEYSTORE);

    configuration.setKeyStore(resourceToFile("clientKeystore"));
    assertThat(configuration.getKeyStore(), nullValue());
  }

  private File createDefaultProtocolConfigFile() throws IOException {
    return createConfigFile(DEFAULT_SECURITY_MODEL, format("defaultProtocol=%s", SUPPORTED_PROTOCOL));
  }

  private File createDefaultConfigFile() throws IOException {
    String contents = format("enabledCipherSuites=UNSUPPORTED,%s\n" + "enabledProtocols=UNSUPPORTED,%s",
                             SUPPORTED_CIPHER_SUITE, SUPPORTED_PROTOCOL);

    return createConfigFile(DEFAULT_SECURITY_MODEL, contents);
  }

  private File createConfigFile(String securityModel, String contents) throws IOException {
    String path = getClassPathRoot(getClass()).getPath();
    File file = new File(path, format(PROPERTIES_FILE_PATTERN, securityModel));

    PrintWriter writer = new PrintWriter(file, "UTF-8");
    writer.println(contents);
    writer.close();

    return file;
  }

  // this is a shorter version of the snippet from:
  // http://www.davidflanagan.com/blog/2005_06.html#000060
  // (see comments; DF's "manual" version works fine too)
  public static URL getClassPathRoot(Class clazz) {
    CodeSource cs = clazz.getProtectionDomain().getCodeSource();
    return (cs != null ? cs.getLocation() : null);
  }

  private String resourceToFile(String resourceName) throws IOException, URISyntaxException {
    return new File(this.getClass().getClassLoader().getResource(resourceName).toURI())
        .getAbsoluteFile()
        .getCanonicalPath();
  }

}
