/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.tooling.internal.deployment.processor;

import static org.mule.runtime.ast.api.util.MuleAstUtils.emptyArtifact;
import static org.mule.runtime.module.deployment.internal.processor.SerializedAstArtifactConfigurationProcessor.serializedAstWithFallbackArtifactConfigurationProcessor;
import static org.mule.runtime.module.tooling.internal.dsl.ArtifactDeclarationUtils.toArtifactast;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.emptySet;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.runtime.app.declaration.serialization.api.ElementDeclarationJsonSerializer;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.config.ConfigurationException;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.runtime.core.api.extension.ExtensionManager;
import org.mule.runtime.deployment.model.api.artifact.ArtifactConfigurationProcessor;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContext;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContextConfiguration;
import org.mule.runtime.dsl.api.ConfigResource;
import org.mule.runtime.module.deployment.internal.processor.AbstractAstConfigurationProcessor;

import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import org.apache.commons.io.IOUtils;

public class SerializedArtifactDeclarationArtifactConfigurationProcessor extends AbstractAstConfigurationProcessor {

  private ArtifactConfigurationProcessor delegate = serializedAstWithFallbackArtifactConfigurationProcessor();

  @Override
  public ArtifactContext createArtifactContext(ArtifactContextConfiguration artifactContextConfiguration)
      throws ConfigurationException {
    if (Stream.of(artifactContextConfiguration.getConfigResources()).noneMatch("serialized-artifact-declaration.decl"::equals)) {
      return delegate.createArtifactContext(artifactContextConfiguration);
    } else {
      return super.createArtifactContext(artifactContextConfiguration);
    }
  }

  @Override
  protected ArtifactAst obtainArtifactAst(ArtifactContextConfiguration artifactContextConfiguration)
      throws ConfigurationException {
    return createApplicationModel(artifactContextConfiguration.getMuleContext(),
                                  artifactContextConfiguration.getConfigResources(),
                                  artifactContextConfiguration.getArtifactProperties(),
                                  artifactContextConfiguration.getArtifactType(),
                                  artifactContextConfiguration.getParentArtifactContext()
                                      .map(ArtifactContext::getArtifactAst)
                                      .orElse(emptyArtifact()),
                                  artifactContextConfiguration.isDisableXmlValidations(),
                                  artifactContextConfiguration.getExpressionLanguageMetadataService());
  }

  private Set<ExtensionModel> getExtensions(ExtensionManager extensionManager) {
    return extensionManager == null ? emptySet() : extensionManager.getExtensions();
  }

  private ArtifactAst createApplicationModel(MuleContext muleContext,
                                             String[] artifactConfigResources,
                                             Map<String, String> artifactProperties,
                                             ArtifactType artifactType,
                                             ArtifactAst parentArtifactAst,
                                             boolean disableXmlValidations,
                                             ExpressionLanguageMetadataService expressionLanguageMetadataService)
      throws ConfigurationException {
    Set<ExtensionModel> extensions = getExtensions(muleContext.getExtensionManager());

    try {
      final var serializedDeclaration =
          IOUtils.toString(new ConfigResource("serialized-artifact-declaration.decl").getInputStream(), UTF_8);
      final var deserializedArtifactDeclaration = ElementDeclarationJsonSerializer.getDefault(true)
          .deserialize(serializedDeclaration, ArtifactDeclaration.class);

      return toArtifactast(deserializedArtifactDeclaration, extensions);
    } catch (Exception e) {
      throw new ConfigurationException(e);
    }
  }

}
