/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static java.util.Optional.of;
import static org.mule.runtime.module.troubleshooting.internal.TroubleshootingTestUtils.mockApplication;
import static org.mule.runtime.module.troubleshooting.internal.TroubleshootingTestUtils.mockDeploymentService;
import static org.mule.runtime.module.troubleshooting.internal.TroubleshootingTestUtils.mockFlowStackEntry;
import static org.mule.runtime.module.troubleshooting.internal.operations.EventDumpOperation.APPLICATION_ARGUMENT_DESCRIPTION;
import static org.mule.runtime.module.troubleshooting.internal.operations.EventDumpOperation.APPLICATION_ARGUMENT_NAME;
import static org.mule.runtime.module.troubleshooting.internal.operations.EventDumpOperation.EVENT_DUMP_OPERATION_DESCRIPTION;
import static org.mule.runtime.module.troubleshooting.internal.operations.EventDumpOperation.EVENT_DUMP_OPERATION_NAME;

import static java.time.Clock.fixed;
import static java.time.Instant.now;
import static java.time.Instant.ofEpochMilli;
import static java.time.ZoneId.of;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonMap;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsIterableWithSize.iterableWithSize;
import static org.hamcrest.CoreMatchers.containsString;

import static org.junit.jupiter.api.Assertions.assertThrows;

import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.mock;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.core.api.context.notification.FlowCallStack;
import org.mule.runtime.core.api.context.notification.FlowStackElement;
import org.mule.runtime.core.api.event.EventContextService;
import org.mule.runtime.core.api.event.EventContextService.FlowStackEntry;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContext;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.troubleshooting.api.ArgumentDefinition;

import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;

import java.io.IOException;
import java.io.StringWriter;
import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.MockedStatic;

public class EventDumpOperationTestCase {

  private DeploymentService deploymentService;
  private EventDumpOperation eventDumpOperation;

  @BeforeEach
  public void setup() {
    FlowStackEntry flowStackEntry = mockFlowStackEntry("001");
    Application app1 = mockApplication("app1",
                                       // shufled to ensure sorting
                                       mockFlowStackEntry("001_z", flowStackEntry),
                                       // an envent context withput hierarchy
                                       mockFlowStackEntry("abc"),
                                       mockFlowStackEntry("001_1", flowStackEntry),
                                       flowStackEntry);
    Application app2 = mockApplication("app2");
    deploymentService = mockDeploymentService(app1, app2);
    eventDumpOperation = new EventDumpOperation(deploymentService);
  }

  @Test
  public void definitionHasCorrectNameDescriptionAndNumberOfArguments() {
    assertThat(eventDumpOperation.getDefinition().getName(), is(EVENT_DUMP_OPERATION_NAME));
    assertThat(eventDumpOperation.getDefinition().getDescription(), is(EVENT_DUMP_OPERATION_DESCRIPTION));
    assertThat(eventDumpOperation.getDefinition().getArgumentDefinitions(), iterableWithSize(1));
  }

  @Test
  public void applicationArgumentDefinitionIsCorrect() {
    ArgumentDefinition applicationArgumentDefinition = eventDumpOperation.getDefinition().getArgumentDefinitions().get(0);
    assertThat(applicationArgumentDefinition.getName(), is(APPLICATION_ARGUMENT_NAME));
    assertThat(applicationArgumentDefinition.getDescription(), is(APPLICATION_ARGUMENT_DESCRIPTION));
    assertThat(applicationArgumentDefinition.isRequired(), is(false));
  }

  @Test
  public void whenNoApplicationIsPassedItReturnsAllApplications() throws IOException {
    final var writer = new StringWriter();
    executeEventDump(emptyMap(), writer);
    String result = writer.toString();

    var expected = """
        Active Events for application 'app1'
        ------------------------------------

        Total Event Contexts:      4
        Total Root Contexts:       2
        Dropped Events:            0
        Dropped Errors:            0


        "001" hierarchy

            "001_1", running for: 00:00.000, state: EXECUTING
                at ns:component@MockLocation(null) 66 ms

            "001_z", running for: 00:00.000, state: EXECUTING
                at ns:component@MockLocation(null) 66 ms

        "001", running for: 00:00.000, state: EXECUTING
            at ns:component@MockLocation(null) 66 ms

        "abc", running for: 00:00.000, state: EXECUTING
            at ns:component@MockLocation(null) 66 ms

        Active Events for application 'app2'
        ------------------------------------

        Total Event Contexts:      0
        Total Root Contexts:       0
        Dropped Events:            0
        Dropped Errors:            0

        """;
    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    String normalizedExpected = expected.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(normalizedExpected)));
  }

  @Test
  public void whenApplicationIsPassedItReturnsOnlyThePassedOne() throws IOException {
    final var writer = new StringWriter();
    executeEventDump(singletonMap(APPLICATION_ARGUMENT_NAME, "app1"), writer);
    String result = writer.toString();

    var expected = """
        Total Event Contexts:      4
        Total Root Contexts:       2
        Dropped Events:            0
        Dropped Errors:            0


        "001" hierarchy

            "001_1", running for: 00:00.000, state: EXECUTING
                at ns:component@MockLocation(null) 66 ms

            "001_z", running for: 00:00.000, state: EXECUTING
                at ns:component@MockLocation(null) 66 ms

        "001", running for: 00:00.000, state: EXECUTING
            at ns:component@MockLocation(null) 66 ms

        "abc", running for: 00:00.000, state: EXECUTING
            at ns:component@MockLocation(null) 66 ms

        """;
    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    String normalizedExpected = expected.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(normalizedExpected)));
  }

  private void executeEventDump(Map<String, String> args, final StringWriter writer) throws IOException {
    Instant instant = now(fixed(ofEpochMilli(66), of("UTC")));
    try (MockedStatic<Instant> mockedStatic = mockStatic(Instant.class)) {
      mockedStatic.when(Instant::now).thenReturn(instant);

      eventDumpOperation.getCallback().execute(args, writer);
    }
  }

  @Test
  public void whenTheEventContextServiceIsNotPresentItRaisesAnException() throws IOException {
    for (Application application : deploymentService.getApplications()) {
      Registry registry = application.getArtifactContext().getRegistry();
      when(registry.lookupByName(EventContextService.REGISTRY_KEY)).thenReturn(empty());
    }

    Map<String, String> arguments = new HashMap<>();
    arguments.put(APPLICATION_ARGUMENT_NAME, "app1");
    final var writer = new StringWriter();
    assertThrows(IllegalArgumentException.class, () -> executeEventDump(arguments, writer));
  }

  @Test
  public void whenEventsAreDroppedTheyAreMarkedInTheDump() throws IOException {
    FlowStackEntry droppedFlowStackEntry = mockFlowStackEntry("dropped_event");
    when(droppedFlowStackEntry.getDroppedAt()).thenReturn(of(ofEpochMilli(100)));

    Application appWithDroppedEvents = mockApplication("appWithDroppedEvents", droppedFlowStackEntry);

    // Mock the EventContextService to return dropped event IDs
    EventContextService eventContextService = mock(EventContextService.class);
    when(eventContextService.getCurrentlyActiveFlowStacks()).thenReturn(singletonList(droppedFlowStackEntry));
    when(eventContextService.getDroppedEventIds()).thenReturn(Set.of("dropped_event"));
    when(eventContextService.getDroppedErrorIds()).thenReturn(Set.of());

    Registry registry = mock(Registry.class);
    when(registry.lookupByName(EventContextService.REGISTRY_KEY)).thenReturn(of(eventContextService));

    ArtifactContext artifactContext = mock(ArtifactContext.class);
    when(artifactContext.getRegistry()).thenReturn(registry);
    when(appWithDroppedEvents.getArtifactContext()).thenReturn(artifactContext);

    deploymentService = mockDeploymentService(appWithDroppedEvents);
    eventDumpOperation = new EventDumpOperation(deploymentService);

    final var writer = new StringWriter();
    executeEventDump(singletonMap(APPLICATION_ARGUMENT_NAME, "appWithDroppedEvents"), writer);

    String output = writer.toString();
    assertThat(output, containsString("Total Event Contexts:      1"));
    assertThat(output, containsString("Total Root Contexts:       1"));
    assertThat(output, containsString("Dropped Events:            1"));
    assertThat(output, containsString("Dropped Errors:            0"));
    assertThat(output, containsString("[DROPPED]"));
    assertThat(output, containsString("(dropped at:"));
  }

  @Test
  public void whenErrorsAreDroppedTheyAreMarkedInTheDump() throws IOException {
    FlowStackEntry droppedErrorFlowStackEntry = mockFlowStackEntry("dropped_error_event");
    when(droppedErrorFlowStackEntry.getDroppedErrorAt()).thenReturn(of(ofEpochMilli(200)));

    Application appWithDroppedErrors = mockApplication("appWithDroppedErrors", droppedErrorFlowStackEntry);

    // Mock the EventContextService to return dropped error IDs
    EventContextService eventContextService = mock(EventContextService.class);
    when(eventContextService.getCurrentlyActiveFlowStacks()).thenReturn(singletonList(droppedErrorFlowStackEntry));
    when(eventContextService.getDroppedEventIds()).thenReturn(Set.of());
    when(eventContextService.getDroppedErrorIds()).thenReturn(Set.of("dropped_error_event"));

    Registry registry = mock(Registry.class);
    when(registry.lookupByName(EventContextService.REGISTRY_KEY)).thenReturn(of(eventContextService));

    ArtifactContext artifactContext = mock(ArtifactContext.class);
    when(artifactContext.getRegistry()).thenReturn(registry);
    when(appWithDroppedErrors.getArtifactContext()).thenReturn(artifactContext);

    deploymentService = mockDeploymentService(appWithDroppedErrors);
    eventDumpOperation = new EventDumpOperation(deploymentService);

    final var writer = new StringWriter();
    executeEventDump(singletonMap(APPLICATION_ARGUMENT_NAME, "appWithDroppedErrors"), writer);

    String output = writer.toString();
    assertThat(output, containsString("Total Event Contexts:      1"));
    assertThat(output, containsString("Total Root Contexts:       1"));
    assertThat(output, containsString("Dropped Events:            0"));
    assertThat(output, containsString("Dropped Errors:            1"));
    assertThat(output, containsString("[ERROR_DROPPED]"));
    assertThat(output, containsString("(error dropped at:"));
  }

  @Test
  public void whenChildStackHasParentOnlyUniqueElementsAreShown() throws IOException {
    // Create parent and child FlowStackEntry
    FlowStackEntry parentEntry = mockFlowStackEntry("parent_event");
    FlowStackEntry childEntry = mockFlowStackEntry("child_event", parentEntry);

    // Get the FlowStackElement from parent's stack (it has "MockLocation")
    FlowStackElement parentElement = parentEntry.getFlowCallStack().getElements().get(0);

    // Create child element with location "child" (reusing same ComponentIdentifier pattern as parent)
    final var tci = TypedComponentIdentifier.builder()
        .identifier(buildFromStringRepresentation("ns:component"))
        .type(OPERATION)
        .build();
    ComponentLocation childLocation = mock(ComponentLocation.class);
    when(childLocation.getComponentIdentifier()).thenReturn(tci);
    when(childLocation.getLocation()).thenReturn("child");

    // Create child element (need to mock Instant.now() for constructor)
    FlowStackElement childElement;
    Instant fixedInstant = ofEpochMilli(0);
    try (MockedStatic<Instant> mockedStatic = mockStatic(Instant.class)) {
      mockedStatic.when(Instant::now).thenReturn(fixedInstant);
      childElement = new FlowStackElement("ChildFlow", "child", childLocation, emptyMap());
    }

    // Replace parent stack (using the element from mockFlowStackEntry, which has "MockLocation")
    FlowCallStack parentStack = mock(FlowCallStack.class);
    when(parentStack.getElements()).thenReturn(List.of(parentElement));
    when(parentEntry.getFlowCallStack()).thenReturn(parentStack);

    // Replace child stack with child element + parent element (child stacks clone parent and push new)
    FlowCallStack childStack = mock(FlowCallStack.class);
    when(childStack.getElements()).thenReturn(List.of(childElement, parentElement));
    when(childEntry.getFlowCallStack()).thenReturn(childStack);

    // Create application with parent and child events
    Application appWithHierarchy = mockApplication("appWithHierarchy", parentEntry, childEntry);

    // Mock EventContextService to return both entries
    EventContextService eventContextService = mock(EventContextService.class);
    when(eventContextService.getCurrentlyActiveFlowStacks()).thenReturn(List.of(parentEntry, childEntry));

    Registry registry = mock(Registry.class);
    when(registry.lookupByName(EventContextService.REGISTRY_KEY)).thenReturn(of(eventContextService));

    ArtifactContext artifactContext = mock(ArtifactContext.class);
    when(artifactContext.getRegistry()).thenReturn(registry);
    when(appWithHierarchy.getArtifactContext()).thenReturn(artifactContext);

    deploymentService = mockDeploymentService(appWithHierarchy);
    eventDumpOperation = new EventDumpOperation(deploymentService);

    final var writer = new StringWriter();
    executeEventDump(singletonMap(APPLICATION_ARGUMENT_NAME, "appWithHierarchy"), writer);
    String result = writer.toString();

    var expected = """
        Total Event Contexts:      2
        Total Root Contexts:       1
        Dropped Events:            0
        Dropped Errors:            0


        "parent_event" hierarchy

            "child_event", running for: 00:00.000, state: EXECUTING
                at ns:component@child(null) 66 ms

        "parent_event", running for: 00:00.000, state: EXECUTING
            at ns:component@MockLocation(null) 66 ms

        """;
    // Normalize line endings to handle differences between Windows (\r\n) and Unix (\n)
    String normalizedResult = result.replaceAll("\\r\\n", "\n");
    String normalizedExpected = expected.replaceAll("\\r\\n", "\n");
    assertThat(normalizedResult, is(equalTo(normalizedExpected)));
  }
}
